package edu.uprm.walsaip.vte.core.renderer;

import java.nio.FloatBuffer;

import javax.media.opengl.GL;

import edu.uprm.walsaip.vte.core.Drawable;
import edu.uprm.walsaip.vte.core.messagebus.MessageListener;
/**
 * Interface for objects which abstract the OpenGL rendering systems. Different implementation used different methods for rendering the Drawable objects.
 * @author Ricardo Veguilla
 * @see Drawable
 */
public interface Renderer extends Drawable, MessageListener {
	/**
	 * Module identifier.
	 */
	public static final String ID = Renderer.class.getSimpleName();
	/**
	 * Message name
	 */
	public static final String SET_POLYGON_TYPE = "setPolygonType";
	/**
	 * Message name
	 */
	public static final String SET_COLOR = "setColor";
	/**
	 * Message name
	 */
	public static final String SET_TEXTURE = "setTexure";
	
	/**
	 * Initialization method.
	 * @param gl Reference to the current GL object.
	 */
	public void init(GL gl);
	/**
	 * Allocates resources for rendering an specifc number of triangles
	 * @param numTriangles number of triangles to be drawn.
	 */
	public void setupBuffers(int numTriangles);

	/**
	 * Render the previously specified geometry with the default rendering options for this renderer module
	 * @param gl Reference to the current GL object.
	 * @return number of triangle rendered
	 */
	public int render(GL gl);
	/**
	 * Render the previously an specific number of previously specified geometry elements using an specific primitive type.
	 * 
	 * @param gl Reference to the current GL object.
	 * @param type primitive type to use
	 * @param numElements number of elements to render.
	 * @return number of triangle rendered
	 */
	public int render(GL gl, int type,int numElements);
	/**
	 * Method to be invoked at the beginning of the rendering of a frame of animation.
	 * @param gl
	 */
	public void render_frame_begin(GL gl);
	/**
	 * Method to be invoked at the end of the rendering of a frame of animation.
	 * @param gl
	 */
	public void render_frame_end(GL gl) ;

	/**
	 * Add a vertex to be rendered.
	 * @param idx index of the vertex
	 * @param xPos x coordinate
	 * @param yPos y coordinate
	 * @param zPos z coordinate
	 * @return index for next element
	 */
	public int addVertex(int idx, float xPos, float yPos, float zPos);
	/**
	 * Add a vertex to be rendered.
	 * @param idx index of the vertex
	 * @param vertex buffer containing the vertex coordinates.
	 * @return index for next element
	 */
	public int addVertex(int idx, FloatBuffer vertex);
	/**
	 * Add a vertex to be rendered.
	 * @param idx index of the vertex
	 * @param vertex array containing the vertex coordinates.
	 * @return index for next element
	 */
	public int addVertex(int idx, float[] vertex);
	
	/**
	 * @return the next available texture id.
	 */
	public int getNextTextureId() ;
	/**
	 * 
	 * @return number of triangles rendered so far.
	 */
	public int numTrianglesRendered();
	
}
