package com.c5corp.c5viewUi;

/**
* A panel of UI widgets and some interface logic for C5 DEM Tool,
* with associated classes InfoTA (a text area to display file info)
* and DEMfileFilter, a subclass of javax.swing.filechooser.FileFilter
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
*/

 /*
 * This library is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free Software
 * Foundation; either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Please refer to LICENSE.txt which is distributed with the distribution for the
 * full text of the GNU Lesser General Public License
*/

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.io.*;
import java.util.*;
import com.c5corp.c5dem.*;
import com.c5corp.DEMconvert.*;

/**
 * Viewer display panel contains UI widgets and is the meat of the
 * C5 DEM Tool GUI
 */
public class ViewerDisplayPanel extends JPanel
		implements MouseMotionListener, ActionListener {

	JTextField xt;
	JTextField yt;
	JTextField et;
	JTextField nt;
	JTextField at;
	JTextField ft;

	JButton color;
	JButton info;
	JButton export;
	JButton open;

	DemTable demtable;
	DemImagePanel pan;
	JFrame fr;
	DemTool view;

	// for info dialog
	JDialog info_dialog;
	infoTA infota;

	// for filter select dialog
	// a JFrame for filter selection
	JFrame select;

	// JList object containing filter selection
	JList list;

	// OK button for the filter selection
	JButton ok;
	JButton cancel;
	JButton filter_info;

	/** Constructs a new ViewerDisplayPanel */
	ViewerDisplayPanel(DemTable demtable, DemImagePanel pan, JFrame fr) {
		this.demtable=demtable;
		this.pan=pan;
		this.fr=fr;
		setLayout(new GridLayout(4,2));
		Label x = new Label("X value:");
		xt = new JTextField(3);
		Label y = new Label("Y value:");
		yt = new JTextField(3);
		Label e = new Label("Easting:");
		et = new JTextField(8);
		Label n = new Label("Northing:");
		nt = new JTextField(9);
		Label a = new Label("Elevation (M):");
		at = new JTextField(4);
		Label f = new Label("Elevation (F):");
		ft = new JTextField(4);
		color=new JButton("color");
		color.addActionListener(this);
		info=new JButton("info");
		info.addActionListener(this);
		export=new JButton("export");
		export.addActionListener(this);
		open=new JButton("open");
		open.addActionListener(this);
		infota=new infoTA(demtable);
		// buttons for file filter loading
 		ok = new JButton("OK");
 		cancel = new JButton("cancel");
		filter_info = new JButton("filter info");
		// hook up the OK button
		ok.addActionListener(this);
		filter_info.addActionListener(this);
		cancel.addActionListener(this);
		// other things for loading filters
		Vector<Class> classes_found = new Vector<Class>();
		File file = new File("com/c5corp/DEMconvert/filters");
		String[] arr = file.list(new ClassFileFilter());

		// add the components to the viewer panel (this)
		add(x);
		add(xt);
		add(y);
		add(yt);
		add(e);
		add(et);
		add(n);
		add(nt);
		add(a);
		add(at);
		add(f);
		add(ft);
		add(color);
		add(info);
		add(export);
		add(open);

		// info dialog stuff
		infota.setEditable(false);
		info_dialog = new JDialog(fr, "DEM file info", false);
		info_dialog.setLocation(50,50);
		info_dialog.setSize(400,400);
		info_dialog.getContentPane().add(infota);

		////////////
		// Get the filters

		// look for classes in the com.c5corp.DEMconvert.filters package
		// this is throwing up when non C5DemAbstractFilter classes are in
		// com\c5corp\DEMconvert\filters
		for (int i=0; i<arr.length; i++) {
			try {
				// dynamically load the class
				Class temp= Class.forName("com.c5corp.DEMconvert.filters."
					+ arr[i].substring(0,arr[i].lastIndexOf(".")));
				// add the class to classes_found
				// ignore any classes that do not have the superclass C5DemAbstractFilter
				// all filters must be a direct descendant
				if (temp.getSuperclass().toString().endsWith("C5DemAbstractFilter")) {
					classes_found.addElement(temp);
				}
			} catch(ClassNotFoundException ex) {
				System.err.println("com.c5corp.DEMconvert.filters." +
					arr[i].substring(0,arr[i].lastIndexOf(".")) +
						" could not be dynamically loaded.");
			} catch (Exception ex) {
				System.err.println("com.c5corp.DEMconvert.filters." +
						arr[i].substring(0,arr[i].lastIndexOf(".")) +
						" " + ex.toString());
			}
		}

		// add filters to a JList
		Enumeration enumer = classes_found.elements();
		String[] filter_names = new String[classes_found.size()];
		int k=0;
		while (enumer.hasMoreElements()) {
			filter_names[k] = ((Class)enumer.nextElement()).getName();
			filter_names[k] = filter_names[k].substring(filter_names[k].lastIndexOf(".")+1,filter_names[k].length());
			k++;
		}

		list=new JList(filter_names);

		// add the panel to a scroll pane
		JScrollPane scroll = new JScrollPane(list);

		// create a sub panel for the buttons
		JPanel buttons = new JPanel();
		buttons.add(filter_info);
		buttons.add(cancel);
		buttons.add(ok);

		// add the panel to a frame
		select = new JFrame();
		select.getContentPane().setLayout(new BorderLayout());
		select.setLocation(50,100);
		select.setSize(250,300);
		select.getContentPane().add(scroll, BorderLayout.CENTER);
		select.getContentPane().add(buttons, BorderLayout.SOUTH);
		select.setTitle("Select Filter");
	}

    /**
     * Tracks the movement of the mouse over the image and changes the corresponding
     *    Easting, Northing, and elevation text desplayed
     */
    public void mouseMoved(MouseEvent e) {
		UtmCoordinatePairElev temp= demtable.getPoint((short)e.getX(),(short)e.getY());
		xt.setText(Integer.toString(e.getX()));
		yt.setText(Integer.toString(e.getY()));
		if (temp == null) {
			et.setText("?");
			nt.setText("?");
			at.setText("?");
			ft.setText("?");
			return;
		}
		et.setText(Double.toString(temp.getEasting()));
		nt.setText(Double.toString(temp.getNorthing()));
		at.setText(Integer.toString(temp.getElevation()));
		ft.setText(Integer.toString((int)(temp.getElevation()*C5DemConstants.METERSTOFEET)));
	}

    /**
     * Forward the mouseDragged event to the underlying child container.
     *   does nothing in this applet
     */
    public void mouseDragged(MouseEvent e) {
		// do nothing
	}

    /**
     * Event Handler
     *   deals with the grayscale/color, export, info, open, "ok" buttons
     */
    public void actionPerformed(ActionEvent e) {
		if (pan.isColor() && e.getActionCommand().equals("grayscale")) {
			pan.showGrayScale();
			color.setText("color");
		} else if (!pan.isColor() && e.getActionCommand().equals("color")) {
			pan.showColor();
			color.setText("grayscale");
		} else if (e.getActionCommand().equals("export")) {
			openFilterDialog();
		} else if (e.getActionCommand().equals("info")) {
			openInfoDialog();
		} else if (e.getActionCommand().equals("open")) {
			loadNewFile();
		} else if (e.getActionCommand().equals("OK")) {
			outputFile(true);
		} else if (e.getActionCommand().equals("filter info")) {
			C5DemAbstractFilter filter = loadFilter((String)list.getSelectedValue());
			JOptionPane.showMessageDialog(null, filter.getFilterInfo());
		} else { // the filter selector was canceled
			outputFile(false);
		}
	}

    /**
     * Loads a new DEM file
     */
    private void loadNewFile() {
		// get the file object
		File file = DemTool.openFile();

		// if the file is null, the user canceled the JFileDialog - see DemTool.openFile()
		// so just return from here and go on like nothing happened
		if (file == null) return;

		// create the dem table
		demtable= new DemTable(file);

		// make a DemImagePanel
		if (pan.isColor()) {
			pan = new DemImagePanel(demtable, DemImagePanel.COLOR);
		} else {
			pan = new DemImagePanel(demtable, DemImagePanel.GRAYSCALE);
		}

		// make a new info text area...
		infota = new infoTA(demtable);

		// make a scrollpane - (just in case the image is really big
		JScrollPane scrollpane = new JScrollPane(pan);
		scrollpane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED );
		scrollpane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED );

		// clear the frame's contentPane
		fr.getContentPane().removeAll();

		// add new stuff to the Jframe
		fr.getContentPane().setLayout(new BorderLayout());
		fr.setLocation(0,0);
		fr.getContentPane().add(scrollpane, BorderLayout.CENTER);
		fr.getContentPane().add(this, BorderLayout.SOUTH);
		fr.setTitle("C5 DEM Tool: " + demtable.get_file_name());
		fr.validate();

		// change the info_dialog
		infota=new infoTA(demtable);
		info_dialog.getContentPane().removeAll();
		info_dialog.getContentPane().add(infota);

		pan.addMouseMotionListener(this);

	}

    /**
     * Loads a new c5demOutFilterInterface object when user selects export
     */
    private void openFilterDialog() {
		// show the frame
		select.setVisible(true);
	}

    /**
     * Outputs the file according to the selected filter in select.
     */
    private void outputFile(boolean ok) {

		// hide the selection frame
		select.setVisible(false);

		C5DemAbstractFilter out_filter;

		if (ok) {
			out_filter = loadFilter((String)list.getSelectedValue());
			String path = out_filter.getOutputPath() + "/";
			PrintWriter out;

			// create printwriter with public static PrintWriter openOutputFile(String out_filename)
			// of the out_filter object
			out = out_filter.openOutputFile(demtable.get_file_name(), path);

			// write the header
			out_filter.writeHeader(demtable, out);

			// write the data
			out_filter.writeData(demtable, out);

			// clean up
			out.close();

			// inform the user
			JOptionPane.showMessageDialog(null, "File Saved in " + path + " directory");

		} else {
			select.setVisible(false);
		}
	}

    /**
     * Helper method creates a filter object from the class files in com/c5corp/DEMconvert/filters/
     */
    private C5DemAbstractFilter loadFilter(String filter_name) {
		// declarations
		Class temp;

		try {
			temp= Class.forName("com.c5corp.DEMconvert.filters." + filter_name);
		} catch(ClassNotFoundException ex) {
			System.out.println("com.c5corp.DEMconvert.filters." + filter_name +
					" could not be dynamically loaded.");
			JOptionPane.showMessageDialog(null, "Select a filter");
			return null;
		} catch (Exception ex) {
			System.out.println("com.c5corp.DEMconvert.filters." + filter_name +
					" " + ex.toString());
			JOptionPane.showMessageDialog(null, "com.c5corp.DEMconvert.filters." +
				filter_name +
					" " + ex.toString());
			return null;
		}

		// C5DemAbstractFilter filter object
		try {
			return (C5DemAbstractFilter)temp.newInstance();
		} catch (InstantiationException e) {
			System.out.println(e);
			JOptionPane.showMessageDialog(null, "com.c5corp.DEMconvert.filters." +
				filter_name +
					" " + e.toString());
			return null;
		} catch (IllegalAccessException e) {
			System.out.println(e);
			JOptionPane.showMessageDialog(null, "com.c5corp.DEMconvert.filters." +
				filter_name +
					" " + e.toString());
			return null;
		}
	}

    /**
     * Overrides java.lang.Object.toString()
     */
    public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}

    /**
     * Opens pop up info dialogue box for corresponding DEM file when info button is pressed
     */
    private void openInfoDialog() {
		infota.updateText();
		info_dialog.setVisible(true);
	}
}

/**
 * panel for the info dialog
 * implements C5DemConstants for some of the variables,
 * system[], units[], vdatum[] and hdatum[]
 */
class infoTA extends JTextArea implements C5DemConstants {
	Dem dem;

	infoTA(Dem dem) {
		this.dem=dem;
		setLineWrap(true);
		setWrapStyleWord(true);
		updateText();
	}

    /**
     * Updates info box text
     */
    public void updateText() {
		// Data Element 1
		String str=new String();
		setText(str);
		str += "File info:\n";
		str += dem.get_file_name() + "\n";
		// Data Element 3
		str += "DEM level code:\t\t" + Short.toString(dem.get_dem_level_code())+ "\n";
		// Data Element 5
		str += "planimetric system:\t\t" +
			system[dem.get_planimetric_system()] + "\n";
		// Data Element 6
		str += "planimetric zone:\t\t" + Short.toString(dem.get_planimetric_zone()) + "\n";
		// Data Element 8
		str += "planametic unit:\t\t" + units[dem.get_planimetric_unit()-1] + "\n";
		// Data Element 9
		str += "elevation unit:\t\t" + units[dem.get_elevation_unit()-1] + "\n";
		// Data Element 12
		str += "min elevation:\t\t" + Double.toString(dem.get_min_elevation()) + "\n";
		str += "max elevation:\t\t" + Double.toString(dem.get_max_elevation()) + "\n";
		// Data Element 15
		str += "spacial units x:\t\t" + Double.toString(dem.get_spacial_rez_x()) + "\n";
		str += "spacial units y:\t\t" + Double.toString(dem.get_spacial_rez_y()) + "\n";
		str += "spacial units z:\t\t" + Double.toString(dem.get_spacial_rez_z()) + "\n";
		// Data Element 16
		str += "columns:\t\t" + Short.toString(dem.get_column_count()) + "\n";
		// undefined by DEM format, calculated in file
		str += "max col size:\t\t" + Short.toString(dem.maxElevationsForAllProfiles()) + "\n";
		// Data Element 26
		str += "vertical datum:\t\t" + vdatum[dem.get_vertical_datum()-1] + "\n";
		// Data Element 27
		str += "horizontal datum:\t\t" + hdatum[dem.get_horizontal_datum()-1] + "\n";

		setText(str);
	}

    /**
     * Overrides java.lang.Object.toString()
     */
    public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}




