package com.c5corp.c5utm.util;

import java.io.*;
import java.util.*;
import java.sql.*;
import com.c5corp.c5dem.*;
import com.c5corp.c5utm.*;

/** <code>public class Dat2C5UTM</code>
* Dat2C5UTM.java - part of the C5 Landscape database (otherwise known as the C5UTM database).
* update the C5UTM database (using java JDBC) from a DEM and a corpscon out
* Dat2C5UTM.java adds UTM points to the C5UTM db, using
* both a DEM (for metadata) and a .out file (a corpscon batch file)
* for the converted points.
* It is necessary when points must have their datum converted.
* refer to sql directory for info on the data model
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class Dat2C5UTM {
	static String messages = ""; // Messages generated by in the processing for RECEIPTS dir
	public static void main(String[] args) {

		// decs
		C5UTMconfs conf = new C5UTMconfs();

		String url = conf.getDbUrl();
		String driver = conf.getDriver();
		//String accessPass = conf.getAccessPassword();
		String updatePass = conf.getUpdatePassword();

		String verticalDatum = conf.getVerticalDatum();
		String horizontalDatum = conf.getHorizontalDatum();
		String verticalUnits = conf.getVerticalUnits();
		String horizontalUnits = conf.getHorizontalUnits();

		// more decs
		String readme="Check out the README.";
      		String batname;
      		String demname;
		String corpscondir;
		String demdir;
		String str="";	// for file input
		BufferedReader corpsconin=null;
		DemTable dem;
		UtmPoint point = new UtmPoint();
		String[][] corners= new String[4][2];
		String demid= "";
		StringTokenizer tokenizer;

		// check for the database before going on.
		if (!(conf.databaseWorking() && conf.databaseUnitsConfigured())) {
			System.err.println("Dat2C5UTM can't start:");
			System.err.println(conf.getMessages());
			return;
		}

      		// get the the corpscon file name to process
		if (args.length >= 2) {
			batname=args[0];
			demname=args[1];
		} else {
			System.err.println("usage: Dat2C5UTM DEM_NAME.out DEM_NAME.dem [CORPSCON dir] [DEM dir]");
			System.err.println(readme);
			return;
		}

		// check the filenames and extentions
		{
			String bat, batext, demf, demext;
			bat=batname.substring(0,batname.lastIndexOf('.'));
			demf=demname.substring(0,demname.lastIndexOf('.'));
			batext=batname.substring(batname.lastIndexOf('.'),batname.length());
			demext=demname.substring(demname.lastIndexOf('.'),demname.length());
			if (!(demext.equals(".dem") && batext.equals(".out"))) {
				System.err.println("usage: first arg must be a CORPSCON .out, the second a .dem");
				System.err.println(readme);
				return;
			}
			if (!(bat.equals(demf))) {
				System.err.println("usage: Dat2C5UTM DEM_NAME.out DEM_NAME.dem; DEM_NAME must match");
				System.err.println("(" + bat + " is not " + demf + ")");
				System.err.println(readme);
				return;
			}
		}

		// check for an alternative path to the corpscon directory (in case it is not under ./CORPSCON
		if (args.length >= 3) {
			corpscondir = args[2];
			if (corpscondir.charAt(corpscondir.length()-1) != '/') corpscondir=corpscondir+"/";
		} else {
			corpscondir= conf.getCorpsconDir() + "/";
		}

		// check for an alternative path to the DEM directory, (in case it is not uder opt/DEM)
		if (args.length >= 4) {
			demdir = args[3];
			if (demdir.charAt(demdir.length()-1) != '/') demdir=demdir+"/";
		} else {
			demdir = conf.getDemDir() + "/";
		}

		// open the CORPSCON file DEM_NAME.out
		try {
			corpsconin = new BufferedReader(new FileReader(corpscondir+batname));
		} catch (FileNotFoundException e) {
			System.err.println("Can't find: " + corpscondir + batname);
			return;
		}

		// create a Dem object from the original source file
     		dem = new DemTable(demdir+demname);

		// read the header data and verify the units and datum
		// check datum and units against the conf units
		try {
			str=corpsconin.readLine();
		} catch (IOException e) {
			System.err.println(e);
		}

		{
			boolean hdat=false;
			boolean horzunits=false;
			boolean vdat=false;
			boolean vertunits=false;
			String error = "The CORPSCON file contains data that is not compatible.\nRequired: ";

			// read
			while (str.charAt(0)==';' || str.charAt(0)=='#') {	// while a comment line
				try {
					str=corpsconin.readLine(); // !-- here above for comment below... line 172ish
					if (str.toUpperCase().indexOf(horizontalDatum.toUpperCase())!=-1 &&
					str.toUpperCase().indexOf("HORIZONTAL DATUM")!=-1) {
						hdat=true;
					} else if (str.toUpperCase().indexOf(horizontalUnits.toUpperCase())!=-1 &&
					str.toUpperCase().indexOf("HORIZONTAL UNITS")!=-1) {
						horzunits=true;
					} else if (str.toUpperCase().indexOf(verticalDatum.toUpperCase())!=-1 &&
					str.toUpperCase().indexOf("VERTICAL DATUM")!=-1) {
						vdat=true;
					} else if (str.toUpperCase().indexOf(verticalUnits.toUpperCase())!=-1 &&
					str.toUpperCase().indexOf("VERTICAL UNITS")!=-1) {
						vertunits=true;
					}
				} catch (IOException e) {
					System.err.println(e);
				}
			}

			// check
			if (!hdat) {
				System.err.println(error + "'Horizontal Datum: UTM, " + horizontalDatum + "'");
				return;
			} else if (!horzunits) {
				System.err.println(error + "'Horizontal Units: " + horizontalUnits +"'");
				return;
			} else if (!vdat) {
				System.err.println(error + "'Vertical Datum: " + verticalDatum + "'");
				return;
			} else if (!vertunits) {
				System.err.println(error + "'Vertical Units: " + verticalUnits + "'");
				return;
			}
		}

		// read the first four points, which are the sw, nw, ne and se corners
		// these should be input as doubles - the first one is still in str
		// not taken from dem of course - these have been converted to the correct datum.
		// there is a record read waiting to be procesed, see // !-- here above line 133ish

		for (int i=0; i<4; i++) {
			/* Check to make sure these points are there! (The CORPSCON out file must be
			created from a .dat created with the correct output filter, so that the first
			four points represent data element 11 of the original dem file, which are
			the ground coordinates of the four corners of the map. The four records should
			look something like this:

			sw,     584891.629179, 4469859.807092,
			nw,     584734.538979, 4483734.976166,
			ne,     595326.560235, 4483862.555371,
			se,     595503.292658, 4469987.296089,

			They should normally be in this order (because the only output filter
			so far does things this way...). Checking that each line contains an
			appropriate case. Good enough... unless some rouge filter somday reorders...
			*/
			if (!(str.startsWith("sw") || str.startsWith("nw")
					|| str.startsWith("ne") || str.startsWith("se"))) {
				System.err.println("Can not find sw, nw, ne, or se corner in data.");
				System.err.println("The input file must be derived from an appropriate filter.");
				return;
			}

			try {
				int index = -1; // this will cause an IndexOutOfBounds exception if
						// we do not find sw, nw, ne, or se
				String temp;
				// make the tokenizer
				tokenizer = new StringTokenizer(str, ",");
				temp = tokenizer.nextToken(); // read the corner value sw, nw, ne, se
				if (str.length() > 2) temp = temp.substring(0,2);
				if (temp.equals("sw")) {
					index = 0;
				} else if (temp.equals("nw")) {
					index = 1;
				} else if (temp.equals("ne")) {
					index = 2;
				} else if (temp.equals("se")) {
					index = 3;
				}
				//corners[index][0]= new Double(Double.valueOf(tokenizer.nextToken())).doubleValue());
				//corners[index][1] = new Double(Double.valueOf(tokenizer.nextToken())).doubleValue());
				corners[index][0]= tokenizer.nextToken();
				corners[index][1] = tokenizer.nextToken();
			} catch (NumberFormatException e) {
				System.err.println("problem with corners data.");
				System.err.println("The input file must be derived from an appropriate filter.");
				System.err.println(e);
				return;
			} catch (IndexOutOfBoundsException e) {
				System.err.println("Can not find sw, nw, ne, or se corner in data.");
				System.err.println("The input file must be derived from an appropriate filter.");
				System.err.println(e);
				return;
			}

			if (i==3) break; // I don't want the next point read yet...

			try {
				str=corpsconin.readLine();
			} catch (IOException e) {
				System.err.println(e);
			}
		}

		/*
		// test corners
		for (int i=0; i<4; i++) {
			for (int j=0; j<2; j++) {
				System.out.println(corners[i][j]);
			}
		}
		*/

		// If we get this far, everything should be ready to go
      		// add the data to the db

                try {
                        Class.forName(driver);
                } catch (Exception e) {
                        System.err.println("Failed to make DB connection. Run verifydb");
                        return;
                }

                try {
			// check that the DEM is not already in the DB
			// exit with and error message if it is	SHINN MOUNTAIN, CA-24000
			String testname = dem.get_file_name();
			testname = testname.substring(0,testname.lastIndexOf(",")+12); // example: SHINN MOUNTAIN, CA-24000
			System.out.println("Checking if '" + testname + "' already exists in the database...");

			/* // this job was given to the
                        Connection connection=DriverManager.getConnection(url, "C5UTM_access", accessPass);
                        Statement select = connection.createStatement();
                        ResultSet result = select.executeQuery
				("SELECT file_info FROM DEM_METADATA where file_info LIKE '" + testname + "%'");
			*/

			Vector vec = C5UTM.findExactDemNames(testname, DbHelper.getDbConnection());

			if (vec.size() == 0) { // if there is no match, we are OK, go ahead
				print("Does not exist in DB, (OK), adding: ");
				print(dem.get_file_name());
			} else { // refuse to add - exit out
				print("Can not add, already in DB: ");
				//println(result.getString("file_info"));
				println(((DemMetadata)(vec.get(0))).getFileInfo());
				return;
			}

                        //select.close();
                        //connection.close();     // close db connection
                } catch (Exception e) {
					System.err.println("Exception when checking for pre-existing entry.");
                    e.printStackTrace();
					return;
                }

			try {
            	Connection connection=DriverManager.getConnection(url, "C5UTM_update", updatePass);
				Statement insert = connection.createStatement();

			// insert a new DEM_METADATA row (use dem object accessors etc.)

			// dem_id		SMALLINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY
			// AUTO_INCREMENT will do the work for dem_id
			// zone			TINYINT
			// columns_x		SMALLINT
			// max_profile_length_y	SMALLINT
			// sw_easting		FLOAT
			// sw_northing		FLOAT
			// nw_easting		FLOAT
			// nw_northing		FLOAT
			// ne_easting		FLOAT
			// ne_northing		FLOAT
			// se_easting		FLOAT
			// se_northing		FLOAT
			// date_added		DATETIME
			// local_dem_file	CHAR(40)
			// file_info		CHAR(140)

			// candiate for additoon to C5UTM
			insert.executeUpdate("INSERT INTO DEM_METADATA (zone, columns_x, " +
				"max_profile_length_y, sw_easting, sw_northing, " +
				"nw_easting, nw_northing, ne_easting, ne_northing, se_easting, se_northing, " +
				"date_added, local_dem_file, file_info) " +
				"values (" +
				dem.get_planimetric_zone() + ", " +
				dem.get_column_count() + ", " +
				dem.maxElevationsForAllProfiles() + ", " +
				corners[0][0] + ", " +
				corners[0][1] + ", " +
				corners[1][0] + ", " +
				corners[1][1] + ", " +
				corners[2][0] + ", " +
				corners[2][1] + ", " +
				corners[3][0] + ", " +
				corners[3][1] + ", " +
				"NOW()" + ", " +
				"'" + demname + "', " +
				"'" + dem.get_file_name() +
				"')"
			);

                        insert.close();
                        connection.close();     // close db connection
			println("\nAdded new DEM_METADATA record...");
                } catch (Exception e) {
					System.err.println("Exception on adding new DEM_METADATA\n");
                    e.printStackTrace();
					return;
                }

			// get the new DEM_METADATA.id
            try {

			Vector vec = C5UTM.findExactDemNames(dem.get_file_name(), conf, DbHelper.getDbConnection());

			// there should only ever be one unless there has
			// been an error adding records to the database

			if (vec.size() < 1) {
				System.err.println("This should never happen.\n" +
					"This code on checks the new DEM_METADATA record that\n" +
					"was just added. Perhaps the access pasword is wrong?\n" +
					"Anyway, serious bad. Good luck fixing.\n");
			} else {
				for (int i=0; i < vec.size(); ++i) {
					demid = ((DemMetadata)(vec.get(0))).getDemId();
					println("Was able to retrieve new DEM_METADATA.dem_id.");
					println("The dem_id is: " + demid);
				}
			}
			} catch (Exception e) {
				System.err.println("Exception on checking the new DEM_METADATA record that\n"
					+	"was just added. Serious problem. Try manually deleting that record.");
							e.printStackTrace();
				return;
			}

		// ok, add the points.
		// Candidate for C5UTM
		try {
			Connection connection=DriverManager.getConnection(url, "C5UTM_update", updatePass);

			// read the first point
			point = readUTMdata(corpsconin, point);
			int count=0;

			while(point != null) { // point should be returned null from readUTMdata at eof
				count++;
				// add the points to UTM_COORDS from the corpscon file
				//

				// id			INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY
				// dem_id		SMALLINT UNSIGNED NOT NULL
				// easting		MEDIUMINT UNSIGNED
				// northing		MEDIUMINT UNSIGNED
				// elevation		SMALLINT
                       		Statement insert = connection.createStatement();
                        	insert.executeQuery("INSERT INTO UTM_COORDS " +
					"(dem_id, easting, northing, elevation)" +
					"values (" +
					demid + "," +
					point.easting + "," +
					point.northing + "," +
					point.elevation + ")"
				);
                        	insert.close();
				// read next point
				point = readUTMdata(corpsconin, point);
			}
			println("Added " + count + " points to UTM_COORDS with dem_id " + demid + ".");
            connection.close();     // close db connection
		} catch (Exception e) {
			System.err.println("Exception on adding new UTM_COORDS, delete dem_id: " + demid + "\n");
			e.printStackTrace();
		}

		println("Writing receipt file.");
		ReceiptWriter.writeReceipt(
			conf,
			dem,
			"Dat2C5UTM",
			messages,
			demid
		);

	}

	// removes values from str, converts to float and rounds to in, returns it all in the UtmPoint
	private static UtmPoint readUTMdata(BufferedReader in, UtmPoint point) {
		String str;
		StringTokenizer tokenizer;
		try {
			str=in.readLine();

			// tokenize the string - each field has 4 tokens
			if (str != null) {
				tokenizer= new StringTokenizer(str, ",");
				tokenizer.nextToken(); // toss out the first token
				point.easting = Math.round(Float.valueOf(tokenizer.nextToken()).floatValue());
				point.northing = Math.round(Float.valueOf(tokenizer.nextToken()).floatValue());
				point.elevation = Math.round(Float.valueOf(tokenizer.nextToken()).floatValue());
			} else {
				return null;
			}

		} catch (IOException e) {
			System.err.println(e);
		}
		return point;
	}

	// the following static functions mirror print and println of System.out
	// but allow certain info to be sent to the screen and to a receipt file.
	private static void print(String str) {
		System.out.print(str);
		messages = messages + str;
	}

	private static void println(String str) {
		System.out.println(str);
		messages = messages + str + "\n";
	}
}

