package com.c5corp.c5utm.util;

/**
* This utility fills the UTM_1K_GRID_DATA table in the C5
* landscape database from data in the DEM_METADATA and UTM_COORS tables.
* (The UTM_POINT_STATS table and MODE are populated as well.)
* (The actual records are stored in then UTM_POINT_STATS table.)
* This program is semi-smart. It sets the DEM_METADATA.1K_grid
* attibute to 1 (partially processed) if the area is successfully processed,
* but there is missing data around the edges due to the lack of surrounding
* elevations from adjacent dems that are needed to calculate
* the area completely. If the attribute is already set to 1,
* the algorithm attempts to recalculate only the
* edges of these areas on subsequent runs. (Build1KGridFromFiles also processes
* the attribute to level 1.) If this script can determine that the entire dem
* has been processed, then it updates the DEM_METADATA.1K_grid attribute to 2,
* and will no longer try to process that area. The DEM_METADATA.1K_grid
* attribute is 0 after new dems have been added to the database.
* @see Build1KGridFromFiles
* @see Build1KGridStats
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0.3
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import com.c5corp.c5utm.*;
import java.sql.*;
import java.util.Vector;
import java.io.File;

public class Build1KGridFromDb {

	private Connection connection = null;
	private boolean verbose = false;
	private C5UTMconfs confs = null;

	/**
	* Construction processes UTM_COORDS data into UTM_1K_GRID_DATA and UTM_POINT_STATS and MODE tables.
	* @param verbose writes messages to System.out if true.
	*/
	public Build1KGridFromDb(boolean verbose) {
		Connection dbh = DbHelper.getDbConnection();
		confs = new C5UTMconfs();
		this.verbose = verbose;

		try {
			// get the DEM_METADATA dem_id and corners for all of the dems in the database which
			// have voids around the edges (1K_grid == 1) or where it is 0 (unprocessed)

			Statement sth = null;
			String sql = null;
			ResultSet result_set= null;
			Vector<DemMetadata> dem_meta_vec = new Vector<DemMetadata>();

			sth = dbh.createStatement();

			sql = "SELECT DEM_METADATA_id, zone, columns_x, max_profile_length_y, sw_easting, " +
				"sw_northing, nw_easting, nw_northing, ne_easting, " +
				"ne_northing, se_easting, se_northing, date_added, " +
				"local_dem_file, file_info, " +
				"1K_grid, point_stats, average_elevation " +
				"FROM DEM_METADATA where (1K_grid=0 || 1K_grid=1) " +
				"ORDER BY zone, sw_easting, sw_northing";

			result_set= sth.executeQuery(sql);

			while (result_set.next()) {
				dem_meta_vec.add(
					new DemMetadata(
						result_set.getString("DEM_METADATA_id"),
						result_set.getInt("zone"),
						result_set.getInt("columns_x"),
						result_set.getInt("max_profile_length_y"),
						result_set.getDouble("sw_easting"),
						result_set.getDouble("sw_northing"),
						result_set.getDouble("nw_easting"),
						result_set.getDouble("nw_northing"),
						result_set.getDouble("ne_easting"),
						result_set.getDouble("ne_northing"),
						result_set.getDouble("se_easting"),
						result_set.getDouble("se_northing"),
						result_set.getDate("date_added"),
						result_set.getString("local_dem_file"),
						result_set.getString("file_info"),
						result_set.getInt("1K_grid"),
						result_set.getInt("point_stats"),
						result_set.getInt("average_elevation")
					)
				);
			}

			sth.close();
			dbh.close();

			// create the update connection db - this one is stored as an instance variable
			// for reuse.
			connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());

			// for each dem in the database, iterate over the 1K divisable grid and update the table
			for(int i = 0; i < dem_meta_vec.size(); i++) {

				DemMetadata utm_meta = (DemMetadata)dem_meta_vec.get(i);

				int result = -1;

				// decide whether to process the edges only (if get_1K_grid_status is 1)
				// else process the whole dem
				
				if (utm_meta.get1kGridStatus() == 1) {
					result = processEdges(utm_meta, connection);
				} else if (utm_meta.get1kGridStatus() == 0) { // points stats status is 0, meaning we should process the whole area
					result = processArea(utm_meta, connection);
				}

				String dem_id = utm_meta.getDemId();

				// update the DEM_METADATA record for
				sql = "UPDATE DEM_METADATA SET 1K_grid=" + result + " WHERE DEM_METADATA_id=\"" + dem_id + "\"";

				sth = connection.createStatement();

				sth.executeUpdate(sql);

				System.out.println("\n*** dem_id" + utm_meta.getDemId() +
					" file:" + utm_meta.getLocalDemFile() + "\nDEM_METADATA.1Kgrid processed to level " +
					result + " on " + new java.util.Date()  + " ***\n");
			}

		} catch (SQLException e) {
			System.err.println(e);
		}
	}

	// iterates over 1K grid area for a dem (represented by ob) - uses insert_record method to add any
	// legitmate areas (full area - no voids) contained in the area.
	// return values:
	// undef should never happen - serious logic error
	// 1 indicates that the area was processed, but there were void
	// areas near the edges (probably becasue points from adjoining dems were not present.)
	// 2 indcates that every point has been processed.
	private int processArea(DemMetadata utm_meta, Connection connection) {
		// !-- This section of code chooses a conservative (slightly larger) area than the dem
		// to iterate over, rounded down (sw corner) or up (ne corner).

		// determine min easting, sw corner
		double start_e = (utm_meta.getSWeasting() < utm_meta.getNWeasting())?utm_meta.getSWeasting():utm_meta.getNWeasting();
		// determine min northing, sw corner
		double start_n = (utm_meta.getSWnorthing() < utm_meta.getSEnorthing())?utm_meta.getSWnorthing():utm_meta.getSEnorthing();
		// determine max easting, ne corner
		double finish_e = (utm_meta.getNEeasting() > utm_meta.getSEeasting())?utm_meta.getNEeasting():utm_meta.getSEeasting();
		// determin max northing ne corner
		double finish_n = (utm_meta.getNWnorthing() > utm_meta.getNEnorthing())?utm_meta.getNWnorthing():utm_meta.getNEnorthing();

		// --! The increase in the oversweep of the areas is to ensure that only dems with every adjacent
		// dem surrounding them is counted as a 2 (completely processed...)
		return run_area_loop(utm_meta, connection, (int)start_e - 500 , (int)start_n - 500, (int)finish_e + 500, (int)finish_n + 500);
	}

	// similar to process_area, except intended for use on DEM_METADATA records
	// that have been partially processed (edges have not been processed because
	// of lacking data from adjacent dems.) Rather than reprocess every point,
	// this function just tries to cover the edges.
	// 1 indicates that the area was processed, but there were void
	// areas near the edges (probably becasue points from adjoining dems were not present.)
	// 2 indcates that every point has been processed.
	private int processEdges(DemMetadata utm_meta, Connection connection) {

		double start_e;
		double start_n;
		double finish_e;
		double finish_n;
		int[] results = new int[4];

		// determine and process west border area between sw and nw corners
		start_e = (utm_meta.getSWeasting() < utm_meta.getNWeasting())?utm_meta.getSWeasting():utm_meta.getNWeasting();
		start_n = (utm_meta.getSWnorthing() < utm_meta.getNWnorthing())?utm_meta.getSWnorthing():utm_meta.getNWnorthing();
		finish_e = (utm_meta.getSWeasting() > utm_meta.getNWeasting())?utm_meta.getSWeasting():utm_meta.getNWeasting();
		finish_n = (utm_meta.getSWnorthing() > utm_meta.getNWnorthing())?utm_meta.getSWnorthing():utm_meta.getNWnorthing();

		// --! The increase in the oversweep of the areas is to ensure that only dems with every adjacent
		// dem surrounding them is counted as a 2 (completely processed...)

		results[0] = run_area_loop(utm_meta, connection, (int)start_e - 500 , (int)start_n - 500, (int)finish_e + 500, (int)finish_n + 500);

		// determine determine and process north border area between nw and ne corners
		start_e = (utm_meta.getNWeasting() < utm_meta.getNEeasting())?utm_meta.getNWeasting():utm_meta.getNEeasting();
		start_n = (utm_meta.getNWnorthing() < utm_meta.getNEnorthing())?utm_meta.getNWnorthing():utm_meta.getNEnorthing();
		finish_e = (utm_meta.getNWeasting() > utm_meta.getNEeasting())?utm_meta.getNWeasting():utm_meta.getNEeasting();
		finish_n = (utm_meta.getNWnorthing() > utm_meta.getNEnorthing())?utm_meta.getNWnorthing():utm_meta.getNEnorthing();

		results[1] = run_area_loop(utm_meta, connection, (int)start_e - 500 , (int)start_n - 500, (int)finish_e + 500, (int)finish_n + 500);

		// determine determine and process east border area between ne and se corners
		start_e = (utm_meta.getNEeasting() < utm_meta.getSEeasting())?utm_meta.getNEeasting():utm_meta.getSEeasting();
		start_n = (utm_meta.getNEnorthing() < utm_meta.getSEnorthing())?utm_meta.getNEnorthing():utm_meta.getSEnorthing();
		finish_e = (utm_meta.getNEeasting() > utm_meta.getSEeasting())?utm_meta.getNEeasting():utm_meta.getSEeasting();
		finish_n = (utm_meta.getNEnorthing() > utm_meta.getSEnorthing())?utm_meta.getNEnorthing():utm_meta.getSEnorthing();

		results[2] = run_area_loop(utm_meta, connection, (int)start_e - 500 , (int)start_n - 500, (int)finish_e + 500, (int)finish_n + 500);

		// determine determine and process south border area between se and sw corners
		start_e = (utm_meta.getSEeasting() < utm_meta.getSWeasting())?utm_meta.getSEeasting():utm_meta.getSWeasting();
		start_n = (utm_meta.getSEnorthing() < utm_meta.getSWnorthing())?utm_meta.getSEnorthing():utm_meta.getSWnorthing();
		finish_e = (utm_meta.getSEeasting() > utm_meta.getSWeasting())?utm_meta.getSEeasting():utm_meta.getSWeasting();
		finish_n = (utm_meta.getSEnorthing() > utm_meta.getSWnorthing())?utm_meta.getSEnorthing():utm_meta.getSWnorthing();

		results[3] = run_area_loop(utm_meta, connection, (int)start_e - 500 , (int)start_n - 500, (int)finish_e + 500, (int)finish_n + 500);

		// return 1 if there is one
		for (int i = 0; i < results.length; i++) {
			if (results[i] == 1) {
				return 1; // unfinished biz in that area
			}
		}
		// found no ones above, they will all be 2 (run_area_loop returns 1 or 2)

		return 2;
	}

	// generalized nested loop for &process_edges and &process_area
	// 1 indicates that the area was processed, but there were void
	// areas near the edges (probably becasue points from adjoining dems were not present.)
	// 2 indcates that every point in the dem has been processed.
	private int run_area_loop(DemMetadata utm_meta, Connection connection,
		int start_e, int start_n, int finish_e, int finish_n) {

		int result=-1;
		int point_result;

		// truncate starts to (min) multiple of 1000
		start_e = (start_e/1000)*1000;
		start_n = (start_n/1000)*1000;

		// truncate finishes to (max) multiple of 1000
		finish_e = (finish_e/1000)*1000;
		finish_e += 1000;
		finish_n = (finish_n/1000)*1000;
		finish_n += 1000;

		// for loops to process area
		// easting loop
		for (int i=start_e; i <= finish_e; i+=1000) {
			for (int j=start_n; j <= finish_n; j+=1000) { // northing loop
				point_result = insert_record(utm_meta.getZone(),i,j, connection);
				// we only care if there were any void areas - indicating a
				// missing adjacent file.
				if (point_result < 2) {
					result = 1;
				}
			}
		}

		if (result == 1) {
			return result;
		} else {
			return 2;
		}
	}

	// This function uses C5UTM.pm to create area, derive stats, and update the db.
	// The return values are listed below
	// 0 indicates that the sw coner of the area being processed was not in the db
	// 1 indicates that the point was there, but the area contained void space (no adjacent points),
	// from areas near the edges.
	// 2 indcates that the point and associated area have been successfully processed. (Including the
	// special case that they have been previously processed.)
	private int insert_record(int zone, int easting, int northing, Connection connection) {

		int result = -1 ;


		try {
			// get an update connection

			if (connection == null) {
				connection = DriverManager.getConnection(confs.getDbUrl(),
					"C5UTM_update", confs.getUpdatePassword());
			}

			Points area = C5UTM.getPoints(zone, easting, northing, 1000, 1000, connection);

			// get the DEM_METADATA dem_id and corners for all of the dems in the database
			if (area != null) {
				if (area.getPercentVoid() > 5) { // if there are any highly void areas
					if (verbose) System.out.println(zone + " " + easting + " " + northing + " is over 5% void, not adding...");
					result = 1;
				} else {
					// see if a matching area exists in UTM_1K_GRID_DATA
					String id = area.getPointAt(0,0).getId(); // checking for sw corner existing in UTM_1K_GRID_DATA
					// check for pre-existing entry in UTM_1K_GRID_DATA
					String sql = "SELECT UTM_1K_GRID_DATA_id FROM UTM_1K_GRID_DATA where UTM_1K_GRID_DATA_id=\"" + id + "\"";
					Statement sth = connection.createStatement();
					ResultSet resultSet = sth.executeQuery(sql);
					// there will only be one record if it exists
					if (resultSet.next()) { // if there is already a record, report it
						if (verbose) System.out.println(zone + " " + easting + " " + northing + " stats already exists in UTM_1K_GRID_DATA");
					} else { // prepare insert the data into the UTM_POINT_STATS and UTM_1K_GRID_DATA tables

						sth.close(); // close the test statement

						String pstat_id = null;
						try {
							pstat_id = area.getPointAt(16,16).getId(); // the id of the middle coordinate
						} catch (ArrayIndexOutOfBoundsException e) {
							if (verbose) System.out.println(zone + " " + easting + " " + northing + " has no points in the center.");
							return 0;
						}

						// test for a pre-existing entry in UTM_POINT_STATS
						sql = "SELECT UTM_POINT_STATS_id FROM UTM_POINT_STATS where UTM_POINT_STATS_id=\"" + pstat_id + "\"";
						sth = connection.createStatement();
						resultSet = sth.executeQuery(sql);
						// there will only be one record if it exists
						if (resultSet.next()) { // report that there is already a UTM_POINT_STATS record, and link it.
							if (verbose) System.out.println(zone + " " + easting + " " + northing + " stats already exists in UTM_POINT_STATS, linking it to UTM_1K_GRID_DATA.");
							sth.close(); // close the test statement
							sql = "INSERT INTO UTM_1K_GRID_DATA (UTM_1K_GRID_DATA_id, UTM_POINT_STATS_id) VALUES (\"" + id + "\", \"" + pstat_id + "\")";
							sth =connection.createStatement();
							sth.executeUpdate(sql);
						} else { // add them both

							sth.close(); // close last statement

							int low = area.getLowest();
							int high = area.getHighest();
							int mean = (int)(area.calculateClosestMean()+.05);
							int median = area.calculateClosestMedian();
							double stdev = area.calculateStandardDeviation();
							int percentile = area.calculatePercentile(area.getPointAt(16,16).getElevation());
							int cmp = (int)(area.calculateContiguousModalityPercentage()+.5); // round to nearest integer
							int topo_descrip = area.calculateTopographyDescriptor();
							int terrain_descrip = area.calculateTerrainDescriptor();
							int[] mode = area.calculateMode();

							if (verbose) {
								System.out.println("***Adding " + zone + " " + easting + " " + northing  + " stats to database***");
								System.out.print(id + ", " + low + ", " + high + ", " + mean + ", " + median + ", ");
								for (int i = 0; i < mode.length-1; i++) {
									System.out.print(mode[i] + " ");
								}
								System.out.print(mode[mode.length-1]);
								System.out.println("), " + stdev + ", " + cmp + ", " + topo_descrip + ", " + terrain_descrip);
							}

							// insert the related UTM_POINT_STATS
							sql = "INSERT INTO UTM_POINT_STATS (UTM_POINT_STATS_id, low, high, mean, median, " +
									"stdev, percentile, contiguous_modality_percentage, " +
									"topographic_descriptor, terrain_descriptor) VALUES(\"" +
								pstat_id + "\", " + low + ", " + high + ", " + mean + ", " +
								median + ", " + stdev + ", " + percentile + ", " + cmp + ", " +
								topo_descrip + ", " + terrain_descrip+ ")";

							sth = connection.createStatement();
							sth.executeUpdate(sql);

							sth.close(); // close last statement

							// insert the related UTM_1K_GRID_DATA
							sql = "INSERT INTO UTM_1K_GRID_DATA (UTM_1K_GRID_DATA_id, UTM_POINT_STATS_id) VALUES (\"" + id + "\", \"" + pstat_id + "\")";
							sth = connection.createStatement();
							sth.executeUpdate(sql);

							// insert the modes for the related area into the MODE table

							for (int i = 0; i < mode.length; i++) {
								sth.close(); // close last statement
								sql = "INSERT INTO MODE (UTM_POINT_STATS_id, mode) VALUES (\"" + pstat_id + "\", \"" + mode[i]+ "\")";
								sth = connection.createStatement();
								sth.executeUpdate(sql);
							}
						}

						sth.close(); // finish either the test statement or insert statement, depending.
						// get the new UTM_POINT_STATS id number and add a record to UTM_1K_GRID_DATA
						// referencing that point

					}

					result = 2;

					// make the image
					createImage(area, zone, easting, northing);
				}
			} else {
				if (verbose) System.out.println("There are no points at " + zone + ", " + easting + ", " + northing);
				// result = 0;
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return result;
	}

	private void createImage(Points points, int zone, int easting, int northing) {

		//build a file name
		String east = new String("" + easting).substring(0,3);
		String north = new String("" + northing).substring(0,4);

		String fileName = new String(zone + "_" + east + "_" + north);

		// create file and check if the file exists
		File file = new File(confs.getUTMImageDir() + File.separator + fileName);
		if (!file.exists()) {
			// write the file
 			UtmImage image = new UtmImage(points,
				UtmImage.RENDER_HIGH |
				UtmImage.RENDER_LOW |
				UtmImage.RENDER_MEAN |
				UtmImage.RENDER_MEDIAN |
				UtmImage.RENDER_MODE
			);

			image.writeImageFile(confs.getUTMImageDir() , fileName);
		}
	}

	/** Run from the command line */
	public static void main (String[] args) {
		try {
			boolean verbose = false;
			if (args[0] != null && args[0].equals("true")) {
				verbose = true;
			}
			new Build1KGridFromDb(verbose);
		} catch (ArrayIndexOutOfBoundsException e) {
			System.err.println("usage: Build1KGridFromDb [true|false]");
			System.err.println("(\"true\" turns on verbose mode)");
		}
	}

	/** Run from another class. Verbose if true argument. */
	public static void Build1KGrid(boolean verbose) {
		new Build1KGridFromDb(verbose);
	}

	/** Simply releases the Connection object. */
	public void finalize() {
		DbHelper.closeDbConnection(connection);
	}

}

