package com.c5corp.c5utm;

/** <code></code>
* <p></p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0.3
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import java.awt.image.BufferedImage;
import java.util.*;
import java.io.*;
import javax.imageio.*;
import javax.imageio.stream.*;
import java.awt.*;

public class UtmImage extends BufferedImage {

	/** If this class has this RenderFlags set, it will render the highest elevations in the image.*/
	public static final int RENDER_HIGH = 64;

	/** If this class has this RenderFlags set, it will render the lowest elevations in the image.*/
	public static final int RENDER_LOW = 32;

	/** If this class has this RenderFlags set, it will render the mean elevations in the image.*/
	public static final int RENDER_MEAN = 16;

	/** If this class has this RenderFlags set, it will render the median elevations in the image.*/
	public static final int RENDER_MEDIAN = 8;

	/** If this class has this RenderFlags set, it will render the modal elevations in the image.*/
	public static final int RENDER_MODE = 4;

	/* A white collar will be rendered only if the constructor
	UtmImage(Points thePoints, int RenderFlags, boolean render_collar).*/
	public static final int RENDER_COLLAR = 2;

	/** If this class has this RenderFlags set, it will simply render an empty collar and utm key, but
	* not elevation (Point) data, even if the other flags are set.
	*/
	public static final int RENDER_EMPTY_COLLAR = 1;

	/* If this class has this RenderFlags set, it will render the a plain grayscale representation
	* of elevation (Point) data. This is the default state of the flags. It is set by the non-collar
	* constructors.
	*/
	private static final int RENDER_PLAIN = 0;

	private Points thePoints = null;
	private int width = 0;
	private int height = 0;
	// private int colorsAllocated = 0; // hang over from the days of gif
	private int renderFlags = RENDER_PLAIN; // default
	private Point[][] pointArray = null;
	private boolean render_collar = false; 	// the UtmImage(Points thePoints, int RenderFlags, boolean render_collar)
											// constructor will turn this on if render_collar is true.

	/** Creates a UtmImage object from a Points object, using default RENDER_PLAIN rendering
	* by calling renderSimple().
	*/
	public UtmImage(Points thePoints) {
		super(thePoints.getPointsArray().length, thePoints.getPointsArray()[0].length, BufferedImage.TYPE_INT_RGB);
		this.thePoints=thePoints;
		pointArray = thePoints.getPointsArray();
		renderSimple();
	}

	/** Creates a UtmImage object from a Points object, with the set RenderFlags. See the
	* public static final ints in this class for a description of what the various
	* render flags do. Calls renderStats().
	*/
	public UtmImage(Points thePoints, int renderFlags) {
		super(thePoints.getPointsArray().length, thePoints.getPointsArray()[0].length, BufferedImage.TYPE_INT_RGB);
		this.thePoints=thePoints;
		pointArray = thePoints.getPointsArray();
		this.renderFlags=renderFlags;
		renderStats();
	}

	/** Creates collared UtmImage object (with a stats legend and UTM tic marks)
	* from a Points object, with the set RenderFlags. See the
	* public static final ints in this class for a description of what the various
	* render flags do. Calls renderStats().
	*/
	public UtmImage(Points thePoints, int renderFlags, boolean renderCollar) {
		super((renderCollar ? thePoints.getPointsArray().length+50 : thePoints.getPointsArray().length),
			(renderCollar ? thePoints.getPointsArray()[0].length+130 : thePoints.getPointsArray()[0].length),
			BufferedImage.TYPE_INT_RGB
		);

		this.renderFlags=renderFlags;
		this.thePoints=thePoints;
		pointArray = thePoints.getPointsArray();
		this.render_collar = renderCollar;
		renderStats();
	}


	/* Returns the number of colors that were allocated during rendering. */
	/*public int getColorsAllocated() {
		return colorsAllocated;
	}*/ // hang over from the days of gif - no need to deprecate, this nor the perl imager was never released

	/** Performs basic rendering of the Points object passed into the constructor.
	* In other words, has the same behavior as if the object is called with
	* RenderFlags equalt to RENDER_PLAIN.
	*/
	public void renderSimple() {
		// local vars needed
		// Shorts (elevations) are the keys, java.awt.Color will be value...
		HashMap<Short,Color> color = new HashMap<Short,Color>();
		HashMap freq_hash = thePoints.calculateFrequencyHash();
		Iterator iter = freq_hash.keySet().iterator();
		Short[] elevations_allocate = new Short[freq_hash.size()];

		int i = 0;

		while(iter.hasNext()) {
			elevations_allocate[i] = (Short)iter.next();
			i++;
		}
		iter = null;
		// sort it
		Arrays.sort(elevations_allocate);

		Short[] points = new Short[elevations_allocate.length];
		for (i = 0; i < elevations_allocate.length; i++) {
			points[i] = elevations_allocate[i];
		}

		Color void_c = null;
		double multiplier;
		int gray=-1;
		int[] color_levels = null;

		int high = thePoints.getHighest();
		int low = thePoints.getLowest();
		int[] shape = thePoints.getArrayShape();
		int largest_y = thePoints.getLargestColumn();
		int the_x = shape.length;
		int elevation_diff = high - low;

		// Allocate points to be mapped to colors if there are more that 128 elevations.
		// 128 grays is our target for downsampling.
		// We infill the remainder of the points in the color hash later to map to these if needed.
		if (elevations_allocate.length > 128) {
			double offset = elevations_allocate.length / (double)128;
			Short[] temp = null;
			Vector<Short> vec = new Vector<Short>();
			for(i=0; i < 128; i++) {
				vec.add(elevations_allocate[(int)(i*offset)]);
			}
			temp = new Short[vec.size()];
			for(i=0; i < vec.size(); i++) {
				temp[i] = (Short)vec.get(i);
			}
			elevations_allocate = temp;
		}

		// if there are voids, allocate the void color, fill in background
		if (thePoints.hasVoidArea()) {
			void_c = new Color(255,110,25); //orange
			//colorsAllocated++;
			// fill it in
			Graphics g = getGraphics();
			g.setColor(void_c);
			g.fillRect(0, 0, the_x, largest_y);
		}

		// determine the multiplier for number of points allocated to create grays
		multiplier = getMultiplier(elevations_allocate.length);

		// create the color levels - with small curve compressing the white spectrum
		// by adjusting the linear scale to a slightly curved one for aesthetic reasons
		color_levels = new int[elevations_allocate.length];
		color_levels[0]=0;
		double adjust = 1.0;
		for(i = 1; i < elevations_allocate.length; i++) {
			color_levels[i] = (int)(i * multiplier * (adjust-=(adjust/256)));
		}

		// readjust to 0-255 scale
		double div = color_levels[color_levels.length-1];
		for (i = 0; i < color_levels.length; i++) {
			color_levels[i] = (int)(color_levels[i]/div*255);
		}

		// make the color table
		for (i=0; i < elevations_allocate.length; i++) {
			gray = color_levels[i];
			color.put(elevations_allocate[i], new Color(gray,gray,gray));
			//colorsAllocated++;
		}

		// infill the color hash if there were more than 128 elevations
		if (points.length > 128) {
			int closest;
			int[] temp = null;
			for (int elev=0; elev < points.length; elev++) {
				if (!color.containsKey(points[elev])) {
					// find the closest allocated color
					temp = new int[elevations_allocate.length];
					for (i = 0; i < temp.length; i++) {
						temp[i] = Math.abs((elevations_allocate[i].intValue())-(points[elev].intValue()));
					}
					int smallest = 100000; // any big value above everest will do
					closest = 0;
					for (i = 0; i < temp.length; i++) {
						if (temp[i] < smallest) {
							smallest = temp[i];
							closest = elevations_allocate[i].intValue();
						}
					}
					color.put(points[elev], color.get(new Short((short)closest)));
				}
			}
		}

		// render myself
		for (int y = largest_y-1; y >= 0; y--) {
			for (int x = 0; x < shape.length; x++) {
				if (thePoints.getPointAt(x, y) != null) {
					// calculate cell color
					int elev = thePoints.getPointAt(x, y).getElevation();
					//$self->{image}->setPixel($x, ($largest_y-$y-1), $color{$elev});
					this.setRGB(x,largest_y-y-1,((Color)color.get(new Short((short)elev))).getRGB() );
				}
			}
		}
	}

	/** Performs basic rendering of the Points object passed into the constructor.
	* I will render statistical info onto the image based on the value of the RenderFlags,
	* which is passed in the constructor. RenderFlags can also be set with setRenderFlags().
	*/
	public void renderStats() {

		// get a graphics object for this Image
		Graphics g = getGraphics();
		Font font = new Font("SansSerif", Font.PLAIN, 9);
		g.setFont(font);

		// booleans
		boolean render_high = ((renderFlags & RENDER_HIGH) == 64);
		boolean render_low = ((renderFlags & RENDER_LOW) == 32);
		boolean render_mean = ((renderFlags & RENDER_MEAN) == 16);
		boolean render_median = ((renderFlags & RENDER_MEDIAN) == 8);
		boolean render_mode = ((renderFlags & RENDER_MODE) == 4);
		boolean render_collar_only = ((renderFlags & RENDER_EMPTY_COLLAR) == 1);

		// count the stats to be rendered
		int total_choices = 0; // only applies to statistics choices, not collar choices
		if (render_high) total_choices++;
		if (render_low) total_choices++;
		if (render_mean) total_choices++;
		if (render_median) total_choices++;
		if (render_mode) total_choices++;

		int i = 0; // just a reuseable iterator var

		HashMap<Short,Color> color = new HashMap<Short,Color>(); // Shorts (elevations) are the keys, java.awt.Color will be value...
		HashMap freq_hash = thePoints.calculateFrequencyHash();
		Iterator iter = freq_hash.keySet().iterator();
		Short[] elevations_allocate = new Short[freq_hash.size()];
		while(iter.hasNext()) {
			elevations_allocate[i] = (Short)iter.next();
			i++;
		}
		iter = null;
		// sort it
		Arrays.sort(elevations_allocate);

		Short[] points = new Short[elevations_allocate.length];
		for (i = 0; i < elevations_allocate.length; i++) {
			points[i] = elevations_allocate[i];
		}

		double multiplier;
		int red=0, green=0, blue=0;

		// every time we render, reset colors_allocated
		//colorsAllocated = 0;

		int mean = thePoints.calculateClosestMean();
		int median = thePoints.calculateClosestMedian();
		int[] mode = thePoints.calculateMode();
		int high = thePoints.getHighest();
		int low = thePoints.getLowest();

		int[] shape = thePoints.getArrayShape();
		int largest_y = thePoints.getLargestColumn();
		int space = 0; // space to be added to largest_y to postion coords and stats on collar
		int the_x = shape.length;
		int elevation_diff = high - low;

		int[] color_levels = null;

		Color void_c = null;
		Color white = null;
		Color ticks = null;

		// if there are voids, allocate the void color.
		if (thePoints.hasVoidArea()) {
			void_c = new Color(255,110,25); //orange
			//colorsAllocated += 1;
		}

		// create a collar if needed
		if (render_collar) {

			// allocate one color for void areas and one for background
			white = new Color(253,254,255); // 'white' for background
			ticks = new Color(25,26,27); // dark gray
			//colorsAllocated+=2;

			// fill the rectangle with white
			g.setColor(white);
			g.fillRect(0, 0, getWidth(), getHeight());

			///// create the grid
			// find the start location of first easting tic
			int xtic=0;
			while (xtic <= 33) { // assuming a 30 meter database - so 33.33 pixels per 1000 meters
				if (thePoints.getPointAt(xtic, largest_y-2) != null) {
					int east = thePoints.getPointAt(xtic, largest_y-2).getEasting();
					int mod = east % 1000;
					if (mod <= 33) break;
				}
				xtic++;
			}

			/// find location of first northing tic
			int ytic=0;
			while (ytic <= 33) { // assuming a 30 meter database - so 33.33 pixels per 1000 meters
				if (thePoints.getPointAt(the_x-2, ytic) != null) {
					int nord = thePoints.getPointAt(the_x-2, ytic).getNorthing();
					int mod = nord % 1000;
					if (mod <= 33) break;
				}
				ytic++;
			}


			// draw the xtics
			g.setColor(ticks);
			for (double x = xtic; x < the_x; x+=33.333333) {
				g.drawLine((int)x,largest_y-1,(int)x,largest_y+6);
				String easting = "void point";
				if (thePoints.getPointAt((int)x, largest_y-2) != null) {
					easting = new String("" + thePoints.getPointAt((int)x, largest_y-2).getEasting());
				}
				g.drawString(easting, (int)x+2, largest_y+14);
			}

			// draw ytics (sometimes, strangely the y ticks are off by one... but it is no big deal...)
			for (double y = ytic; y < largest_y; y+=33.333333) {
				g.drawLine(the_x-1,(int)y,the_x+6,(int)y);
				String northing= "void point";
				if (thePoints.getPointAt(the_x-2,(int)(largest_y-y-1)) != null) {
					northing = new String("" + thePoints.getPointAt(the_x-2,(int)(largest_y-y-1)).getNorthing());
				}
				g.drawString(northing, the_x+8, (int)y);
			}

			// make the font a little bigger
			font = new Font("SansSerif", Font.PLAIN, 11);
			g.setFont(font);

			// add coords to collar
			space=28;
			String coords = new String("Z:" + thePoints.getPointAt(0, 0).getZone() +
				" E:" + thePoints.getPointAt(0, 0).getEasting() +
				" N:" + thePoints.getPointAt(0, 0).getNorthing());

			g.drawString(coords, 5, largest_y+space);

			if (thePoints.hasVoidArea()) {
				//fill in image area with void color - what is not filled in later is void
				g.setColor(void_c);
			} else {
				// fill the background of the dem image plane with white
				g.setColor(white);
			}
			g.fillRect(0, 0, the_x, largest_y);
		}

		// if we are only rendering a collar for another imager, then return the collar now
		if (render_collar_only) return;

		// Allocate points to be mapped to colors if there are more that 128 elevations.
		// 128 grays is our target for downsampling.
		// We infill the remainder of the points in the color hash later to map to these if needed.
		if (elevations_allocate.length > 128) {
			double offset = elevations_allocate.length / (double)128;
			Short[] temp = null;
			Vector<Short> vec = new Vector<Short>();
			for(i=0; i < 128; i++) {
				vec.add(elevations_allocate[(int)(i*offset)]);
			}

			// we may have clobbered important elevations in the smoothing
			// so put them back if needed
			if (render_high) {
				Short sh = new Short((short)high);
				if (!vec.contains(sh)) {
					vec.add(sh);
				}
			}

			if (render_low) {
				Short sh = new Short((short)low);
				if (!vec.contains(sh)) {
					vec.add(sh);
				}
			}

			if (render_mean) {
				Short sh = new Short((short)mean);
				if (!vec.contains(sh)) {
					vec.add(sh);
				}
			}

			if (render_median) {
				Short sh = new Short((short)median);
				if (!vec.contains(sh)) {
					vec.add(sh);
				}
			}

			if (render_mode) {
				for (i=0; i < mode.length; i++) {
					Short sh = new Short((short)mode[i]);
					if (!vec.contains(sh)) {
						vec.add(sh);
					}
				}
			}

			// put them into the temp array
			temp = new Short[vec.size()];
			for(i=0; i < vec.size(); i++) {
				temp[i] = (Short)vec.get(i);
			}

			elevations_allocate = temp;
			Arrays.sort(elevations_allocate);
		}

		// determine the multiplier for number of points allocated to create grays
		multiplier = getMultiplier(elevations_allocate.length);

		// create the color levels - with small curve compressing the white spectrum
		// by adjusting the linear scale to a slightly curved one for aesthetic reasons
		color_levels = new int[elevations_allocate.length];
		color_levels[0]=0;
		double adjust = 1.0;
		for(i = 1; i < elevations_allocate.length; i++) {
			color_levels[i] = (int)(i * multiplier * (adjust-=(adjust/256)));
		}

		// readjust to 0-255 scale
		double div = color_levels[color_levels.length-1];
		for (i = 0; i < color_levels.length; i++) {
			color_levels[i] = (int)(color_levels[i]/div*255);
		}

		// still processing color levels, accounting for stats
		for (i=0; i < elevations_allocate.length; i++) {
			red = color_levels[i];
			green = red;
			blue = red;

			if  (render_mean && elevations_allocate[i].intValue() == mean) { // red (average)
				red = 235;
			}

			if (render_median && elevations_allocate[i].intValue() == median) { // green (median)
				green = 235;
			}

			for (int j=0; j < mode.length; j++) {
				if (render_mode && elevations_allocate[i].intValue() == mode[j]) { // blue (mode[s])
					blue = 235;
				}
			}

			if (render_mean && render_median && render_mode) {
				if ((elevations_allocate[i].intValue() == mean) && (elevations_allocate[i].intValue() == median)) {
					for (int j=0; j < mode.length; j++) {
						if (mode[j] == median) { // also == closest_ave
							red = 8;	// teal - mean == median == mode
							green = 123;
							blue = 123;
						}
					}
				}
			}

			// hot pink is the color for the lowest elevation
			if (render_low && elevations_allocate[i].intValue() == low) {
				red = 255;
				green = 105;
				blue = 180;
			//maroon high is the color for the highest point
			} else if (render_high && elevations_allocate[i].intValue() == high) {
				red = 143;
				green = 0;
				blue = 82;
			}

			// ok, finally
			color.put(elevations_allocate[i], new Color(red,green,blue));
			//colors_allocated++;
		}

		// finish rendering the the collar if needed
		if (render_collar) {;
			space+=16;
			if (render_high) {
				g.setColor((Color)color.get(new Short((short)thePoints.getHighest())));
				g.drawString("High: " + thePoints.getHighest(),5,largest_y+space);
				space+=16;
			}
			if (render_low) {
				g.setColor((Color)color.get(new Short((short)thePoints.getLowest())));
				g.drawString("Low: " + thePoints.getLowest(),5,largest_y+space);
				space+=16;
			}
			if (render_mean) {
				g.setColor((Color)color.get(new Short((short)thePoints.calculateClosestMean())));
				g.drawString("Mean: " + thePoints.calculateClosestMean(),5,largest_y+space);
				space+=16;
			}
			if (render_median) {
				g.setColor((Color)color.get(new Short((short)thePoints.calculateClosestMedian())));
				g.drawString("Median: " + thePoints.calculateClosestMedian(),5,largest_y+space);
				space+=16;
			}

			if (render_mode) {
				String modeString = "";
				for (i=0; i < mode.length; i++) {
					modeString = modeString + mode[i];
				}
				g.setColor((Color)color.get(new Short((short)thePoints.calculateMode()[0])));
				g.drawString("Mode: " + modeString ,5,largest_y+space);
				space+=16;
			}
		}


		// infill the color hash if there were more than (128) elevations
		if (points.length > 128) {
			Short[] temp = null;
			Vector<Short> vec_of_elevations_allocated = new Vector<Short>();
			// create a hash of allocated elevations
			for(i=0; i < elevations_allocate.length; i++) {
				vec_of_elevations_allocated.add(new Short(elevations_allocate[i].shortValue()));
			}

			// if any of the statistics are 'on', we do not want to map other points
			// to them. So we remove them from our allocated
			if (render_high) vec_of_elevations_allocated.remove(new Short((short)high));
			if (render_low) vec_of_elevations_allocated.remove(new Short((short)low));
			if (render_mean) vec_of_elevations_allocated.remove(new Short((short)mean));
			if (render_median) vec_of_elevations_allocated.remove(new Short((short)median));
			if (render_mode) {
				for (i=0; i < mode.length; i++) {
					vec_of_elevations_allocated.remove(new Short((short)mode[i]));
				}
			}

			// now elevations_allocate is clean of statistically special values -
			// so find closest will map to grays only...
			elevations_allocate = new Short[vec_of_elevations_allocated.size()];
			for (i=0; i < elevations_allocate.length; i++) {
				elevations_allocate[i]=(Short)vec_of_elevations_allocated.get(i);
			}

			int smallest;

			for (int elev=0; elev < points.length; elev++) {
				if (!color.containsKey(points[elev])) {
					// find the closest allocated color
					temp = new Short[elevations_allocate.length];
					for (i=0; i < temp.length; i++) {
						temp[i] = new Short((short)Math.abs((elevations_allocate[i].shortValue() - points[elev].shortValue())));
					}

					smallest = 100000; // any big value will do
					int closest = -1; // closest index
					for (i = 0; i < temp.length; i++) {
						if (temp[i].intValue() < smallest) {
							smallest = temp[i].intValue();
							closest = i;
						}
					}
					color.put(points[elev], color.get(elevations_allocate[closest]));
				}
			}
		}

		// render myself
		for (int y = largest_y-1; y >= 0; y--) {
			for (int x = 0; x < shape.length; x++) {
				try {
					if (thePoints.getPointAt(x, y) != null) { // important to check for nulls and array outs
						// calculate cell color
						int elev = thePoints.getPointAt(x, y).getElevation();
						this.setRGB(x,largest_y-y-1,((Color)color.get(new Short((short)elev))).getRGB() );
					}
				} catch (ArrayIndexOutOfBoundsException e) {
					continue;
				}
			}
		}
	}

	/** Sets the rendering flags */
	public void setRenderFlags(int renderFlags) {
		this.renderFlags=renderFlags;
	}

	/** Writes a .png file from this UtmImage object. The path and the file name
	* are given in the parameters. Generally, it is good to use File.separator
	* as your file separator. This method automatically adds File.separator
	* between the path and the file name, and adds the .png extension automatically.
	* @see java.io.File
	*/
	public void writeImageFile(String path, String filename) {
		if (path.equals("") || path == null) {
			writeImageFile(new File(filename + ".png"), "png");
		} else {
			writeImageFile(new File(path + File.separator + filename + ".png"), "png");
		}
	}

	/** Writes a image file from this UtmImage object. The File paramter should
	* be formed with an appropriate extension for the formatName, such as ".png". "png", "jpeg"
	* or other file tyes supported by javax.imageio.ImageIO would be used for the formatName parameter.
	* @see java.io.File
	* @see javax.imageio.ImageIO
	*/
	public void writeImageFile(File file, String formatName) {
		// Get the ImageWrwiter
		Iterator writers = ImageIO.getImageWritersByFormatName(formatName); // png is standard formatName
		ImageWriter writer = (ImageWriter)writers.next();

		ImageOutputStream ios = null;
		try {
			ios = ImageIO.createImageOutputStream(file);
		} catch (IOException e) {
			System.err.println(e);
		}
		writer.setOutput(ios);

		// Finally, the image may be written to the output stream:
		try {
			writer.write(this);
		} catch (IOException e) {

			System.err.println(e);
		}
		writer.dispose();
	}

	// determine the multiplier based on the number of elevations allocated to create grays
	private double getMultiplier(int num) {
		if (num != 0) {
			return (256.0/num);
		} else {
			return 0.0;
		}
	}
}
