package com.c5corp.c5utm;

import java.util.Date;
import java.io.*;
import com.c5corp.c5dem.*;

/** <code>public class Points</code>
* ReceiptWriter.java - part of Landscape database (otherwise known as the C5UTM database).
* the static method writeReceipt of ReceiptWriter creates a receipt file in the RECEIPTS
* directory containing information about any DEM added to the C5UTM db.
* file layout:
<pre>
# datetime;Transaction Date/Time
# fileinfo; Info from the type A record
# demfileinfo;File info
# software;Software adding points to database
# url;URL to Display Map
# backoutid;Backout ID
# backoutcommand;Backout command
</pre>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class ReceiptWriter {
	/**
	<code>public static void writeReceipt(C5UTMconfs conf, Dem dem, String software, String backOutId)</code>
	writes a receipt to the RECEIPT directory configured for the database.
	*/
	public static void writeReceipt(C5UTMconfs conf, DemTable dem, String software, String message, String backOutId) {
		writeReceipt(conf, dem, software, message, backOutId, null);
	}

	/**
	<code>public static void writeReceipt(C5UTMconfs conf, Dem dem, String software, String backOutId, PrintWriter out)</code>
	writes a receipt to the RECEIPT directory. This version accepts a printwriter. It the printwriter is null, it
	creates its own based on the location of the Receipts directory provided in confs.
	*/
	public static void writeReceipt(C5UTMconfs conf, DemTable dem, String software, String message, String backOutId, PrintWriter out) {
		int zone;
		int easting;
		int northing;
		int eastMeters;
		int northMeters;

		// create receipt file name
		String rctfilename = dem.getInputFile();
		rctfilename = rctfilename.substring(rctfilename.lastIndexOf(File.separator),rctfilename.lastIndexOf(".dem"));
		rctfilename = rctfilename + ".dat";

		// get a writer if out is null
		if (out == null) {
			try {
				//System.out.println(conf.getReceiptsDir() + "/" + rctfilename);
				out = new PrintWriter(new FileWriter(conf.getReceiptsDir() + "/" + rctfilename, true), true);
			} catch (IOException e) {
				System.err.println(e);
				return;
			}
		}

		// get the zone
		zone = dem.get_planimetric_zone();

		// find most easterly easting
		if (dem.getSWcorner().getEasting() < dem.getNWcorner().getEasting()) {
			easting = (int)dem.getSWcorner().getEasting();
		} else {
			easting = (int)dem.getNWcorner().getEasting();
		}

		// find most southerly northing
		if (dem.getSWcorner().getNorthing() < dem.getSEcorner().getNorthing()) {
			northing = (int)dem.getSWcorner().getNorthing();
		} else {
			northing = (int)dem.getSEcorner().getNorthing();
		}

		// get the size of the area
		eastMeters = (int)(dem.getSEcorner().getEasting() - dem.getSWcorner().getEasting());
		northMeters = (int)(dem.getNWcorner().getNorthing() - dem.getSWcorner().getNorthing());

		// test the corners...
		//System.out.println("SW easting " + (dem.getSWcorner().getEasting() + " northing " + dem.getSWcorner().getNorthing()));
		//System.out.println("NW easting " + (dem.getNWcorner().getEasting() + " northing " + dem.getNEcorner().getNorthing()));
		//System.out.println("NE easting " + (dem.getNEcorner().getEasting() + " northing " + dem.getNEcorner().getNorthing()));
		//System.out.println("SE easting " + (dem.getSEcorner().getEasting() + " northing " + dem.getSEcorner().getNorthing()));

		// create url - this url will not always work unless the
		// dem to the west of the dem the url represents in also
		// in the database. This is due to the difference between
		// the utm projection not alligning with the paper map edges.
		String url = conf.getBaseUrl() +
			"/cgi-bin/get_map.pl?zone=" +
			zone +
			"&easting=" +
			easting +
			"&northing=" +
			northing +
			"&east_meters=" +
			eastMeters +
			"&north_meters=" +
			northMeters;

		// create back out command
		String backOutCommand = "dem2c5utm -b " + backOutId;

		out.println(
			"datetime;" + new Date().toString() + "\n" +
			"fileinfo;" + dem.get_file_name() + "\n" +
			"demfilename;" + dem.getInputFile() + "\n" +
			"software;" + software  + "\n" +
			"url;" + url + "\n" +
			"backoutid;" + backOutId + "\n" +
			"backoutcommand;" + backOutCommand + "\n" +
			"message;" + message
		);

		out.close();
	}
}
