package com.c5corp.c5utm;

import java.util.*;
import com.c5corp.c5dem.C5DemConstants;
import com.c5corp.c5dem.UtmCoordinatePairElev;

/**
* <p>Points.java - part of Landscape database (otherwise known as the C5UTM database).
* A <code>Points</code> object is designed to hold a 2d array of <code>com.c5corp.c5utm.Point objects</code>,
* and a large collection of accessors and methods that describe or calculate descriptors
* of the area that is represented.<br>
* It provides constructor that you should never call, and accessor/calculator methods for all fields.
* Note that the methods
* prepended "calculate" must perform a calculation of that attribute the first time they are
* called, but will thereafter simply return a cached value. Some of these methods call other calculation
* methods of the same class.<br>
* You would not norally create and instance of this class yourself using new,
* but will instead use the static factory methods
* in the C5UTM class because they do some of the basic calculations for members of this class.
* (In particular, the lowest and highest elevations, the largest_col, building the 2d array from the
* database, and determining if there are void areas with in the requested area.) This factory method approach may seem
* counter intuitive from an OOP point of view, but this mirrors factory subroutines in C5UTM.pm,
* A perl module from which these java classes were spawned.<br>
* This class also provides a println method.<br>
* !!!! this is only partially implemented to match C5UTM.pm - particularly, not all of the calculate and find
* methods of C5UTM are yet implemented.</P>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class Points {

	private int lowest;
	private int highest;
	private int largest_col;
	private Point[][] array;
	private int[] shape;
	private double mean; // calculated if called through accessor
	private int sum; // calculated if called through accessor
	private int count; // calculated if called through accessor
	private double median; // calculated if called through accessor
	private int[] mode; // calculated if called through accessor
	private double standard_dev; // calculated if called through accessor
	private HashMap freq_hash; // calculated if called through accessor
	private int closest_mean; // calculated if called through accessor
	private int closest_median; // calculated if called through accessor
	private int contiguous_modality_percentage; // calculated if called through accessor
	private boolean has_void;
	private double percent_void;
	private byte terrain_descriptor = -2;
	private byte topo_descriptor = -1;

	//
	private boolean mean_calculated;
	private boolean sum_calculated;
	private boolean count_calculated;
	private boolean median_calculated;
	private boolean mode_calculated;
	private boolean std_dev_calculated;
	private boolean freq_hash_calculated;
	private boolean closest_mean_calculated;
	private boolean closest_median_calculated;
	private boolean contiguous_modality_percentage_calculated;

	// Bit masks for use in calculatePercentileDescriptor and calculateTopographyDescriptor
	// Bit order: NW,  W, SW, S, SE, E, NE, N
	// Bit order: 128,64, 32, 16, 8, 4,  2, 1
	//

	/** CARDINAL_BITS, which also include intercardinal primary directions, are arranged declared in the
	* array in the following order: {N_BIT,NE_BIT,E_BIT,SE_BIT,S_BIT,SW_BIT,W_BIT,NW_BIT}. The array
	* CARDINAL_BITS can be used to iterate over the values returned by <code>int calculateTerrainDescriptor()</code>
	* and <code>int calculateTopographyDescriptor()</code> for analysis.
	*/
	public final static int[] CARDINAL_BITS = 	{	C5DemConstants.N_BIT,
													C5DemConstants.NE_BIT,
													C5DemConstants.E_BIT,
													C5DemConstants.SE_BIT,
													C5DemConstants.S_BIT,
													C5DemConstants.SW_BIT,
													C5DemConstants.W_BIT,
													C5DemConstants.NW_BIT
												};

	/** <code>public Points(int zone, int easting, int northing, int elevation, int id)</code>
	* <p>Constructor for the Point class. Normally you would not call this constructor yourself, but
	* would use on of the static methods fo the C5UTM class to create an object of this type.
	* The static methods of the C5UTM class makes an call to the database in which the
	* related data from the UTM_COORDS table is taken and an instance of this class produced.</p>
	*/
	public Points(int lowest, int highest, int largest_col, Point[][] array, int [] shape,
		boolean has_void, double percent_void) {

		if (array == null) {
			throw new NullPointerException();
		}

		this.lowest=lowest;
		this.highest=highest;
		this.largest_col=largest_col;
		this.array=array;
		this.shape=shape;
		this.has_void=has_void;
		this.percent_void=percent_void;

		mean_calculated = false;
		sum_calculated = false;
		count_calculated = false;
		median_calculated = false;
		mode_calculated = false;
		std_dev_calculated = false;
		freq_hash_calculated = false;
		closest_mean_calculated = false;
		closest_median_calculated = false;
		contiguous_modality_percentage_calculated = false;
	}

	/** <code>public int getLowest()</code>
	* <p>Returns the lowest elevation found in the set - useful for imaging</p>
	*/
	public int getLowest() {
		return lowest;
	}

	/** <code>public int getHighest()</code>
	* <p>Returns the highest elevation found in the set - useful for imaging</p>
	*/
	public int getHighest() {
		return highest;
	}

	/**
	* <p>returns the size of the largest column - useful for imaging</p>
	*/
	public int getLargestColumn() {
		return largest_col;
	}

	/**
	* <p>Returns a reference to an array of arrays (a 2d array reference) of com.c5corp.c5utm.Point objects.
	* getArrayShape() returns an array describing each column in the 2d array
	* where the array index represents the 'x' (thus the array length is number of cols),
	* and the length of each col is the value of each index. Point 0,0 is the SW
	* corner, easting to the x, northing to the y. This can be used to iterate over the Point[][].</p>
	* @see Point
	*/
	public Point[][] getPointsArray() {
		return array;
	}

	/**
	* <p>Returns a reference to an array of arrays (a 2d array reference) of com.c5corp.c5dem.UtmCoordinatePairElev objects.
	* This is an older com.c5corp.c5dem type, available here because it has some utility. However, it is
	* preferable to use a Point[][] when possible, because this method converts the Points classes Point[][] object
	* into a UtmCoordinatePairElev[][], throwing away some information.
	* getArrayShape() returns an array describing each column in the 2d array
	* where the array index represents the 'x' (thus the array length is number of cols),
	* and the length of each col is the value of each index. Point 0,0 is the SW
	* corner, easting to the x, northing to the y. This can be used to iterate over the UtmCoordinatePairElev[][].</p>
	* @see #getArrayShape()
	* @see #getArrayShape()
	* @see com.c5corp.c5dem.UtmCoordinatePairElev
	* @see Point
	*/
	public UtmCoordinatePairElev[][] getUtmCoordinatePairElevArray() {
		UtmCoordinatePairElev[][] utmCoordArr = new UtmCoordinatePairElev[shape.length][];
		for (int i = 0; i < shape.length; i++) {
			UtmCoordinatePairElev[] temp = new UtmCoordinatePairElev[shape[i]];
			for (int j = 0; j < temp.length; j++) {
				temp[j] = new UtmCoordinatePairElev(array[i][j].getEasting(), array[i][j].getNorthing(), array[i][j].getElevation());
			}
			utmCoordArr[i]=temp;
		}
		return utmCoordArr;
	}

	/**
	* <p>Returns the point at (X,Y) in the array, or throws an ArrayIndexOutOfBoundsException.
	* It is possible to avoid this exception. See getArrayShape()</p>
	*/
	public Point getPointAt(int x, int y) throws ArrayIndexOutOfBoundsException {
		return array[x][y];
	}

	/**
	* <p>Returns a simple array that can be used to iterate over the
	* 2d array returned by getElevations. Note that the data returned
	* from C5UTM is *not* always a regular x by y grid! (may contain null
	* that represent voids. See also <code>hasVoidArea()</code></p>
	*/
	public int[] getArrayShape() {
		return shape;
	}

	/**
	* <p>This method calculates the mean value of all elevations in the Points object.
	* Once this method has been called and the instance variable is calculated,
	* it simply returns the value stored in the instance variable.</p>
	*/
	public double calculateMean() {
		if (mean_calculated) {
			return mean;
		} else { // calculate it
			int accum = 0;
			int count = 0;

			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { 	// it is important to check nulls
							accum += array[i][j].getElevation();
							count++;
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}
			mean = (double)accum/count;
			mean_calculated = true;
			sum = accum; // save the result so calculateSum does not need to recalculate
			sum_calculated = true;
			this.count = count; // save the count so calculate
			count_calculated = true;
			return mean;
		}
	}

	/**
	* <p>This is a calculation because there may be void areas.
	* This value is also calculated by <code>calculateMean()</code>
	* and <code>calculateSumOfElevations()</code>. If those methods
	* have been called on an object of this class, this value is
	* pre-calculated and will be returned directly from an instance variable.</p>
	*/
	public int calculateNumberOfPoints() {
		if (count_calculated) {
			return this.count;
		} else { // calculate it
			int count = 0;
			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { 	// it is important to check for nulls and array outs
							count++;
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}
			this.count = count; // save the count so calcualte
			count_calculated = true;
			return count;
		}
	}

	/**
	* <p>This method returns the total sume of all of the elevations represented in the Points object.
	* This method is only marginally useful, and an especially problematic way to compare areas
	* if some of the areas have voids. Basically, it only makes sense if the areas have the same
	* number of points, This value is also calculated by <code>calculateMean()</code> If calculateMean()
	* has been called on an object of this class. Once this value has been
	* pre-calculated it will be returned directly from an instance variable.</p>
	*/
	public int calculateSumOfElevations() {
		if (sum_calculated) {
			return this.sum;
		} else { // calculate it
			int sum = 0;
			int count = 0;
			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { 	// it is important to check for nulls and array outs
							sum += array[i][j].getElevation();
							count++; // might as well do this here
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}
			this.count = count; // save the count so calcualte
			count_calculated = true;
			this.sum = sum;
			sum_calculated = true;
			return sum;
		}
	}


	/**
	* <p>Calculates the median value of the elevations. This method calls <code>calculateNumberOfPoints()</code>
	* internally. Once this value has been pre-calculated it will be returned directly from an instance variable.</p>
	*/
	public double calculateMedian() {
		if (median_calculated) {
			return median;
		} else { // calculate it
			// stuff all of the elevations in a flat structure
			int[] arr = new int[this.calculateNumberOfPoints()];
			int count = 0;
			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { // it is important to check for nulls and array outs
							arr[count] = array[i][j].getElevation();
							count++;
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}
			// sort them
			Arrays.sort(arr);
			// find the median value
			if (arr.length % 2 == 0) { // even
				int middle = (arr.length/2);
				median = (arr[middle-1] + arr[middle]) / 2;
			} else { // odd
				median = arr[arr.length/2];
			}
			median_calculated = true;
			return median;

		}
	}

	/**
	* <p>Calculates and returns a java.util.HashMap object. The keys of the hash are Short Objects
	* representing the elevation of a point in the Points array, and the value is an Short representing
	* the number of occurances of the elevation represented by the key value.</p>
	*/
	public HashMap calculateFrequencyHash() {
		if (freq_hash_calculated) {
			return freq_hash;
		} else {
			HashMap<Short,Short> freq_hash = new HashMap<Short,Short>(1550); // good default size for a 34x34 area
			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { // it is important to check for nulls and array outs
							Short elevation = new Short((short)array[i][j].getElevation());
							if (freq_hash.get(elevation) == null) {
								freq_hash.put(elevation, new Short((short)1));
							} else {
								short count = ((Short)freq_hash.get(elevation)).shortValue();
								count++;
								freq_hash.put(elevation, new Short(count));
							}
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}
			this.freq_hash = freq_hash;
			freq_hash_calculated = true;
			return this.freq_hash;
		}
	}

	/**
	* <p>Calculates and returns a java.util.HashMap object. The keys of the hash are Short Objects
	* representing the elevation of a point in the Points array, and the value is an Short representing
	* the number of occurances of the elevation represented by the key value.
	* Once this value has been pre-calculated it will be returned directly from an instance variable.</p>
	*/
	public double calculateStandardDeviation() {
		if (std_dev_calculated) {
			return standard_dev;
		} else {
			double ave = calculateMean();
			double[] arr = new double[this.calculateNumberOfPoints()];
			double accum = 0;
			int count = 0;
			double divided;

			for (int j = largest_col-1; j >=0  ; j--) {
				for (int i = 0; i < shape.length; i++) {
					try {
						if (array[i][j] != null) { // it is important to check for nulls and array outs
							arr[count] = (double)array[i][j].getElevation();
							count++;
						}
					} catch (ArrayIndexOutOfBoundsException e) {
						continue;
					}
				}
			}

			// subtract the mean, square each, and accumulate.
			for (int j = 0; j < arr.length; j++) {
				double val = arr[j];
				val = val - ave;
				val = val * val;
				accum += val;
			}
			divided = accum/(arr.length-1);

			standard_dev = Math.sqrt(divided);
			std_dev_calculated = true;
			return standard_dev;
		}
	}

	/**
	* <p>Calculates and returns an array int[] containing one or more (if multi-modal) modes
	* for the area.</p>
	*/
	public int[] calculateMode() {
		if (mode_calculated) {
			return this.mode;
		} else {
			HashMap hash = calculateFrequencyHash();
			Object[] keys = hash.keySet().toArray();
			Vector<Object> vec = new Vector<Object>();
			int[] mode;
			Short largest = new Short((short)0);
			for (int j = 0; j < keys.length; j++) {
				Short candidate = (Short)hash.get(keys[j]);
				if (candidate.compareTo(largest) == 0) { // if equal to largest
					vec.add(keys[j]);
				} else if (candidate.compareTo(largest) > 0) {
				// if candidate is greater than largest - we have a new mode candidate
					vec.clear();
					largest = candidate;
					vec.add(keys[j]);
				} // else the value at the key was less than the largest so far, and non-modal
			}

			mode = new int[vec.size()];
			for (int j = 0; j < mode.length; j++) {
				mode[j] = ((Short)vec.get(j)).intValue();
			}
			Arrays.sort(mode);
			this.mode = mode;
			mode_calculated = true;
			return mode;

		}
	}

	/**
	* <p>Finds the actual elevation that is member of the Data Set which is closest to the actual mean.</p>
	*/
	public int calculateClosestMean() {
		if (closest_mean_calculated) {
			return closest_mean;
		} else {
			int closest_index = 0;
			double smallest = 100000; // any number higher that everest will do
			double average = calculateMean();
			int i = 0;
			HashMap actual_elevations = calculateFrequencyHash();
			Set keyset = actual_elevations.keySet();
			int[] elevations = new int[keyset.size()];
			Iterator iterator = keyset.iterator();
			// fill the array
			while(iterator.hasNext()) {
				elevations[i] = ((Short)iterator.next()).intValue();
				i++;
			}
			for (i = 0; i < elevations.length; i++) {
				double difference = Math.abs(elevations[i]-average);
				if (difference < smallest) {
					smallest = difference;
					closest_index = i;
				}
			}
			closest_mean_calculated = true;
			closest_mean = elevations[closest_index];
			return closest_mean;
		}
	}

	/**
	* <p>Find the elevation in the Data Set closest to the median.
	* In some situations when there is an even list with two different values in the middle,
	* the median is an average of those two values, usually a
	* fraction and not a member of the set. This method maps to a memeber of the set.
	* This is important when you expect the value to be a member of the set.</p>
	*/
	public int calculateClosestMedian() {
		if (closest_median_calculated) {
			return closest_median;
		} else {
			int closest_index = -1;
			double smallest = 100000; // any number higher that everest will do
			double median = calculateMedian();
			int i=0;
			HashMap actual_elevations = calculateFrequencyHash();
			Set keyset = actual_elevations.keySet();
			int[] elevations = new int[keyset.size()];
			Iterator iterator = keyset.iterator();
			// fill the array
			while(iterator.hasNext()) {
				elevations[i] = ((Short)iterator.next()).intValue();
				i++;
			}

			for (i = 0; i < elevations.length; i++) {
				double difference = Math.abs(elevations[i]-median);
				if (difference < smallest) {
					smallest = difference;
					closest_index = i;
				}
			}
			closest_median_calculated = true;
			closest_median = elevations[closest_index];
			return closest_median;
		}
	}

	/**
	* <p>Calculates the contiguous modality percentage. The cpm is the number of modal elevations that
	* are contiguous with themselves ie, are 'next to" a point with the same (modal) elevation
	* for that area this is to identify how much 'pooling' of the modal elevation(s) there is in the area.</p>
	*/
	public int calculateContiguousModalityPercentage() {
		if (contiguous_modality_percentage_calculated) {
			return contiguous_modality_percentage;
		} else {
			int count = 0;
			int[] mode = calculateMode();
			// this twoD array helps use check the points surrounding a point
			int sur_vals[][] = {{1,-1},{1,0},{1,1},{0,-1},{0,1},{-1,-1},{-1,0},{-1,1}};
			// need to check for nulls here
			for (int x=0; x < shape.length; x++) {
				C1: for (int y = shape[x]-1; y >= 0; y--) {
					// what is the elevation of this point?
					int elev = -4000000;
					if (array[x][y] != null) {
						elev = array[x][y].getElevation();
						// end this iteration unless the point is a modal value
						//next unless grep($elev == $_, @mode);
						for (int b = 0; b < mode.length; b++) {
							if (mode[b] != elev) {
								continue C1;
							}
						}
					}

					// iterate over the points surrounding the point - counting it if there
					// is a an adjacent elevation
					for (int i = 0; i < sur_vals.length; i++) {
						try {
							if (elev == (array[x + sur_vals[i][0]][y + sur_vals[i][1]].getElevation()) ) {
								count++;
								break; // we only care about finding one adjacent elev
							}
						} catch (ArrayIndexOutOfBoundsException e) {
							// found an edge, do nothing
						} catch (NullPointerException e) {
							// another way of finding an edge
							// do nothing
						}
					}
				}
			}
			contiguous_modality_percentage = (int)((double)count/calculateNumberOfPoints()*100);
			contiguous_modality_percentage_calculated = true;
			return contiguous_modality_percentage;
		}
	}


	/**
	* <p>find the Point in the Data Set closest to elevation val. This method does not cache it's result.</p>
	*/
	public int findClosestElevation(int val) {
		HashMap actual_elevations = calculateFrequencyHash();
		Set keyset = actual_elevations.keySet();
		int[] elevations = new int[keyset.size()];
		int i = 0;
		double smallest = 100000; // any number higher that everest will do
		int closest_index = -1;

		Iterator iterator = keyset.iterator();
		// fill the array
		while(iterator.hasNext()) {
			elevations[i] = ((Short)iterator.next()).intValue();
			i++;
		}

		for (i = 0; i < elevations.length; i++) {
			double difference = Math.abs(elevations[i]-val);
			if (difference < smallest) {
				smallest = difference;
				closest_index = i;
			}
		}
		return elevations[closest_index];
	}

	/**
	<p># calculatePercentile  takes one argument, an elevation, and
	returns a value from 0 to 100 representing the percentile
	of that elevation relative to the rest of the set, or -1 or -2<br>
	-1 is a sentinel indicating that the area is flat, thus there is not a percentile to calculate<br>
	-2 is a sentinel indicating that the elevation given is out of the range between highest and lowest<br>
	Note that if it is a flat area, this function returns -1. -2 would infer that there is a range to be out of,
	which there is not given a flat area. Thus NO VALUE will ever return -2 'out of range' for a flat space.
	So you should check for this condition (perfect flatness), before you depend on this function to
	verify that a value is out of the area's range!<br>
	100 indicates the elevation given is == to the highest point in the area<br>
	0 indicates the elevation given is == to the lowest point in the area<br>
	-1, 0, and 100 are useful values for finding flat areas, basins or lowpoints on slopes, and peaks respectively
	if the value is not between the high and the low point in the set.</p>
	*/
	public int calculatePercentile(int elevation) {
		int range = highest - lowest;

		if (range == 0) { // flat area
			return -1;
		} else if (elevation > lowest && elevation < highest) {
			double temp = ((elevation-lowest)/(double)range)*100;
			if (temp < 1) {
				return 1; // for very small factions, return 1 anyway
			} else {
				return (int)temp;
			}
		} else if (elevation == lowest) { // lowest point
			return 0;
		} else if (elevation == highest) {
			return 100;
		} else {
			return -2;
		}
	}

	/**
	* <p>Characterizes the shape of the immediate
	* surrounding area - the immediate points around the point.
	* Each bit refers to a direction, true indicating the terrain is
	* higher in that direction, false indicating that it is equal to
	* or lower terrain in that direction.<br>
	* <tt>Bit order: NW,  W, SW, S, SE, E, NE, N</tt><br>
	* <tt>Bit order: 128,64, 32, 16, 8, 4,  2, 1<br></tt><br>
	* Note that all the range of values will always be 0-255, in spite
	* of this methods return type being an int.
	* The terrain descriptor is thus fairly meaningless as a
	* comparative basis unless you shift through each bit postion
	* in order to find a match (at 0, 45, 90, 135, 180, 225, 270, and 315 degrees.)
	* Returns -1 if the value can not be caluclated.</p>
	*/
	public int calculateTerrainDescriptor() {

		// if already calculated, simply return it
		if (terrain_descriptor != -2) { // -2, default value, see declaration
			return (int)terrain_descriptor;
		}

		int result = 0;
		int[] shape = getArrayShape();
		int size = shape.length;

		// get the x center
		int middle_val_x = get_middle(size);
		int high_val_x = middle_val_x + 1;
		int low_val_x = middle_val_x -1;

		// get the y center
		size = shape[middle_val_x];
		int middle_val_y = get_middle(size);
		int high_val_y = middle_val_y + 1;
		int low_val_y = middle_val_y -1;

		// set up parallel arrays
		int[] x = {middle_val_x,high_val_x,high_val_x,high_val_x,middle_val_x,low_val_x,low_val_x,low_val_x};
		int [] y = {high_val_y,high_val_y,middle_val_y,low_val_y,low_val_y,low_val_y,middle_val_y,high_val_y};

		// iterate over these and build the result
		// array below is the instance var
		try {
			int center_elev =  array[middle_val_x][middle_val_y].getElevation();
			for (int i=0; i < 8; i++) {
				if (array[x[i]][y[i]].getElevation() > center_elev) {
					result = result | CARDINAL_BITS[i];
				}
			}
		} catch (ArrayIndexOutOfBoundsException e) {
			return -1;
		}

		terrain_descriptor = (byte)result;

		return result;
	}

	/**
	* <p>Characterizes the shape of a 1K area -
	* Can be used to identify canyons, draws, spurs, ridges,
	* valleys, depressions and saddles.
	* Each bit refers to a direction, true indicating the terrain is
	* higher in that direction, false indicating that it is equal to
	* or lower terrain in that direction.<br>
	* <tt>Bit order: NW,  W, SW, S, SE, E, NE, N</tt><br>
	* <tt>Bit order: 128,64, 32, 16, 8, 4,  2, 1<br></tt><br>
	* Note that all the range of values will always be 0-255, in spite
	* of this methods return type being an int.
	* The terrain descriptor is thus fairly meaningless as a
	* comparative basis unless you shift through each bit postion
	* in order to find a match (at 0, 45, 90, 135, 180, 225, 270, and 315 degrees.)</p>
	*/
	public int calculateTopographyDescriptor() {

		// if already calculated, simply return it
		if (topo_descriptor != -1) {
			return (int)topo_descriptor;
		}

		int result = 0;
		int[] shape = getArrayShape();
		int size = shape.length;

		// get the x center
		int middle_val_x = get_middle(size);
		int high_val_x = size - 1;

		// get the y center
		int middle_val_y = get_middle(shape[middle_val_x]);
		int nw_high_val_y = shape[0]-1;
		int n_high_val_y = shape[middle_val_x]-1;
		int ne_high_val_y = shape[high_val_x]-1;

		// set up parallel arrays
		int[] x = {middle_val_x,high_val_x,high_val_x,high_val_x,middle_val_x, 0, 0, 0};
		int[] y = {n_high_val_y,ne_high_val_y,middle_val_y, 0, 0, 0,middle_val_y,nw_high_val_y};

		// iterate over these and build the result
		// array below is the instance var

		int center_elev =  array[middle_val_x][middle_val_y].getElevation();
		for (int i=0; i < 8; i++) {
			try {
				while (array[x[i]][y[i]] == null) { // if there are void areas around the edges, can be array outs too
									// back off
					if (x[i] != 0) x[i]--;
					if (y[i] != 0) y[i]--;
				}
				if (array[x[i]][y[i]].getElevation() > center_elev) {
					result = result | CARDINAL_BITS[i];
				}
			} catch (ArrayIndexOutOfBoundsException e) {
				continue;
			}
		}

		topo_descriptor = (byte)result;

		return result;
	}

	/** String representation of the Points object.*/
	public String toString() {
		return
			"SW corner:" +  array[0][0].toString() + "\n" +
			"lowest = " +  lowest + "\n" +
			"highest = " + highest + "\n" +
			"largest col = " + largest_col + "\n" +
			"voids = " + has_void + "\n";
	}

	/** returns a String representation of the south west corner.*/
	public String getSWCorner() {
		return array[0][0].toString();
	}


	// helper to find the middle of the array
       private int get_middle(int num) {

		if ((num % 2) == 0) { 	// even elements
			return (num/2)-1;
		} else {		// odd
			return num/2;
		}
	}

	/** <code>public boolean hasVoidArea()</code>
	* <p>This function returns true if the elevations area has any undefined points.
	* (These would be points not in the database, or near the edges of really large
	* sets where the curvature of the data causes clipping in due to the slight skew
	* in the grid that can occur, (especially if converting points with corpscon...)</p>
	*/
	public boolean hasVoidArea() {
		return has_void;
	}

	/**
	* <p>This function returns the percentage of undefined points to defined points.</p>
	*/
	public double getPercentVoid() {
		return percent_void;
	}
}
