package com.c5corp.c5utm;

import java.util.*;
import java.sql.*;

/**
* <code>public class dbHelper</code>
* DemMetadata.java - part of Landscape database (otherwise known as the C5UTM database).
* The <code>UtmMetaData</code> class is just a structure to hold the
* fields in a UTM_METADATA record. It provides a constructor to initialize the fields
* and accessor methods for all fields. Usually you will not create and instance
* of this class yourself, but will use the static methods in the C5UTMdatabaseMethods
* class to create a UtmMetaData from a query of some sort.
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class DemMetadata {

	// Bit masks for use in int getSurroundingDemAvailability() and
	// DemMetadata getAdjacentDemMetadata(int cardinal_direction);
	// Bit order: NW,  W, SW, S, SE, E, NE, N
	// Bit order: 128,64, 32, 16, 8, 4,  2, 1
	//
	/** Mask to check for NW bit (see CARDINAL_BITS[] below)*/
	public final static int NW_BIT = 128;
	/** Mask to check for W bit (see CARDINAL_BITS[] below)*/
	public final static int W_BIT = 64;
	/** Mask to check for SW bit (see CARDINAL_BITS[] below)*/
	public final static int SW_BIT = 32;
	/** Mask to check for S bit (see CARDINAL_BITS[] below)*/
	public final static int S_BIT = 16;
	/** Mask to check for SE bit (see CARDINAL_BITS[] below)*/
	public final static int SE_BIT = 8;
	/** Mask to check for E bit (see CARDINAL_BITS[] below)*/
	public final static int E_BIT = 4;
	/** Mask to check for NE bit (see CARDINAL_BITS[] below)*/
	public final static int NE_BIT = 2;
	/** Mask to check for N bit (see CARDINAL_BITS[] below)*/
	public final static int N_BIT = 1;
	/** CARDINAL_BITS, which also include intercardinal primary directions, are arranged declared in the
	array in the following order: {N_BIT,NE_BIT,E_BIT,SE_BIT,S_BIT,SW_BIT,W_BIT,NW_BIT}. The array
	CARDINAL_BITS can be used to iterate over the values returned by <code>int getSurroundingDemAvailability().
	*/
	public final static int[] CARDINAL_BITS = {N_BIT,NE_BIT,E_BIT,SE_BIT,S_BIT,SW_BIT,W_BIT,NW_BIT};

	// tracking of the surrounding dems
	private int surroundingDems;
	private boolean surroundingDemsCalculated = false;

	//surrounding demMetadata objects
	private DemMetadata northdem;
	private DemMetadata northeastdem;
	private DemMetadata eastdem;
	private DemMetadata southeastdem;
	private DemMetadata southdem;
	private DemMetadata southwestdem;
	private DemMetadata westdem;
	private DemMetadata northwestdem;

	//grid location in Grid object
	private int gridX;
	private int gridY;

	// changing to a String in verson 1.0.3b to accomodate
	private String dem_id;			//DEM_METADATA_id CHAR(32) NOT NULL PRIMARY KEY,

	//# the UTM zone of the coordinates
	private int zone;			//TINYINT,

	//# the number of profiles in the original DEM - think x (the max val of x in UTM coords)
	private int columns_x;		//SMALLINT,

	//# max_profile_length is not part of the dem specification, but is important
	//# in terms imaging an entire dem. Think max "y" or column value - note that they
	//# can be different sizes because 1:24000 DEM data files use UTM, but the paper maps
	//# are squared on lat/long, so there the column (profile) lengths adjust this skew.
	//# This field counts the number of elevations in the largest profile.
	int max_profile_length_y;	//SMALLINT,

	//# corners
	private double sw_easting;		//FLOAT,
	private double sw_northing;		//FLOAT,
	private double nw_easting;		//FLOAT,
	private double nw_northing;		//FLOAT,
	private double ne_easting;		//FLOAT,
	private double ne_northing;		//FLOAT,
	private double se_easting;		//FLOAT,
	private double se_northing;		//FLOAT,

	//#date_added is not part of the dem specifcation
	//#it holds the date that the DEM was placed in this database.
	private java.sql.Date date_added;		//DATETIME,

	//# local DEM file name
	//# (Path to local DEM file - which is automatically renamed for the geographic name taken from
	//# the first part of the TYPE A record number 1, (first 40 chars), spaces replaced with underscores.
	//# all files must have a .dem extention, which is not recorded here...
	//# note that these files may use *different* horizontal and vertical datums
	//# because all points in C5UTM are converted to NAD 83 and
	private String local_dem_file;		//CHAR(40),

	//# the original file information from the origin DEM, TYPE A record number 1
	private String file_info;		//CHAR(140)

	//# These attributes indicate the status of statistics that have been created for
	//# this dem and stored in the optional tables UTM_1K_GRID_DATA and UTM_POINT_STATS
	//# respectively. See the sql for those tables. Null indicates that the area needs
	//# full processing. 1 indicates that the area was processed, but there were void
	//# areas near the edges (probably becasue points from adjoining dems were not present.)
	//# 2 indcates that every point has been processed.
	private int oneKgrid;			// TINYINT,
	private int point_stats;		// TINYINT,

	private double average_elevation;

	/** Constructs a DemMetadata object, which is normally constructed from data
	* in the database, utilizing static methods as in the C5UTM class.
	*/
	public DemMetadata(String dem_id, int zone,
			int columns_x, int max_profile_length_y,
			double sw_easting, double sw_northing,
			double nw_easting, double nw_northing,
			double ne_easting, double ne_northing,
			double se_easting, double se_northing,
			java.sql.Date date_added, String local_dem_file,
			String file_info, int oneKgrid, int point_stats, double average_elevation) {
		this.dem_id = dem_id;
		this.zone = zone;
		this.columns_x = columns_x;
		this.max_profile_length_y = max_profile_length_y;
		this.sw_easting = sw_easting;
		this.sw_northing = sw_northing;
		this.nw_easting = nw_easting;
		this.nw_northing = nw_northing;
		this.ne_easting = ne_easting;
		this.ne_northing = ne_northing;
		this.se_easting = se_easting;
		this.se_northing = se_northing;
		this.date_added = date_added;
		this.local_dem_file = local_dem_file;
		this.file_info = file_info;
		this.oneKgrid = oneKgrid;
		this.point_stats = point_stats;
		this.average_elevation = average_elevation;

		this.northdem = null;
		this.northeastdem = null;
		this.eastdem = null;
		this.southeastdem = null;
		this.southdem = null;
		this.southwestdem = null;
		this.westdem = null;
		this.northwestdem = null;
		gridX = Integer.MIN_VALUE;
		gridY = Integer.MIN_VALUE;
	}

	/** <p>Returns an integer containing flags describing the surrounding dem availability. If
	* the bit is true, the dem in the represented direction is available in the database. See
	* the public final static int values declared in this class. Note that the class
	* calculates its result only when first called. The answer is cached, and can not be
	* recalculated without getting a new DemMetadata object. Note that this method does not look in
	* adjacent UTM zones yet.</p>
	* <p></p>
	*/
	public int calculateSurroundingDemAvailability(Connection connection) {
		if (surroundingDemsCalculated) {
			return surroundingDems;
		} else {
			// calculate them
			//N
			if (C5UTM.getPoint(zone, (int)(nw_easting+((ne_easting - nw_easting)/2)), (int)(nw_northing + 3000), connection) != null) {
				surroundingDems += N_BIT;
			}
			//NE
			if (C5UTM.getPoint(zone, (int)(ne_easting+3000), (int)(ne_northing+3000), connection) != null) {
				surroundingDems += NE_BIT;
			}
			//E
			if (C5UTM.getPoint(zone, (int)(ne_easting+3000), (int)(ne_northing-((ne_northing - se_northing)/2)), connection) != null) {
				surroundingDems += E_BIT;
			}
			// SE
			if (C5UTM.getPoint(zone, (int)(se_easting+3000), (int)(se_northing-3000), connection) != null) {
				surroundingDems += SE_BIT;
			}
			//S
			if (C5UTM.getPoint(zone, (int)(sw_easting+(se_easting - sw_easting)/2), (int)(sw_northing - 3000), connection) != null) {
				surroundingDems += S_BIT;
			}
			//SW
			if (C5UTM.getPoint(zone, (int)(sw_easting-3000), (int)(sw_northing-3000), connection) != null) {
				surroundingDems += SW_BIT;
			}
			//W
			if (C5UTM.getPoint(zone, (int)(nw_easting-3000), (int)(nw_northing-((nw_northing-sw_northing)/2)), connection) != null) {
				surroundingDems += W_BIT;
			}
			//NW
			if (C5UTM.getPoint(zone, (int)(nw_easting-3000), (int)(nw_northing+3000), connection) != null) {
				surroundingDems += NW_BIT;
			}

			surroundingDemsCalculated = true;
			return surroundingDems;
		}
	}

	/** <p>Returns the DemMetadata object for the defined cardinal direction.
	* See the public final static int values declared in this class for appropriate aguments.
	* Returns 'this' object if one of the static int values defined in this class is not used as an argument, and
	* null if the area does not exist. public int calculateSurroundingDemAvailability(Connection connection)
	* checks for the existence of the surrounding areas. Note that this method does not look in
	* adjacent UTM zones yet.</p>
	@see #calculateSurroundingDemAvailability(Connection connection)
	* <p></p>
	*/
	public DemMetadata getAdjacentDemMetadata(int cardinal_direction, Connection connection) {
		if (cardinal_direction==N_BIT) {//N
			return getDemMetadata((int)(nw_easting+((ne_easting - nw_easting)/2)), (int)(nw_northing + 3000), connection);
		} else if (cardinal_direction == NE_BIT) {//NE
			return getDemMetadata((int)(ne_easting+3000), (int)(ne_northing+3000) , connection);
		} else if (cardinal_direction == E_BIT) {//E
			return getDemMetadata((int)(ne_easting+3000), (int)(ne_northing-((ne_northing - se_northing)/2)), connection);
		} else if (cardinal_direction == SE_BIT) {// SE
			return getDemMetadata((int)(se_easting+3000), (int)(se_northing-3000) , connection);
		} else if (cardinal_direction == S_BIT) {//S
			return getDemMetadata((int)(sw_easting+(se_easting - sw_easting)/2), (int)(sw_northing - 3000), connection);
		} else if (cardinal_direction == SW_BIT) {//SW
			return getDemMetadata((int)(sw_easting-3000), (int)(sw_northing-3000) , connection);
		} else if (cardinal_direction == W_BIT) {//W
			return getDemMetadata((int)(nw_easting-3000), (int)(nw_northing-((nw_northing-sw_northing)/2)), connection);
		} else if (cardinal_direction == NW_BIT) {//NW
			return getDemMetadata((int)(nw_easting- 3000), (int)(nw_northing+3000) , connection);
		} else {
			return this;
		}
	}

	// helper method for getAdjacentDemMetadata
	private DemMetadata getDemMetadata(int easting, int northing, Connection connection) {
		Point point = C5UTM.getPoint(zone, easting, northing, connection);
		return C5UTM.getDemMetadata(point.getDemId() , connection);
	}



	/** sets the reference to the Dem to the north of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setNorthDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffWeasting = this.getNWeasting()-((DemMetadata)listDems.get(i)).getSWeasting();
			diffWeasting = Math.abs(diffWeasting);
			double diffWnorthing = this.getNWnorthing()-((DemMetadata)listDems.get(i)).getSWnorthing();
			diffWnorthing = Math.abs(diffWnorthing);
			double diffEeasting = this.getNEeasting()-((DemMetadata)listDems.get(i)).getSEeasting();
			diffEeasting = Math.abs(diffEeasting);
			double diffEnorthing = this.getNEnorthing()-((DemMetadata)listDems.get(i)).getSEnorthing();
			diffEnorthing = Math.abs(diffEnorthing);
			//if it does exist then add it
			if(diffWeasting < 500 && diffWnorthing < 500 && diffEeasting < 500 && diffEnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				northdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** sets the reference to the Dem to the northeast of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setNortheastDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffeasting = this.getNEeasting()-((DemMetadata)listDems.get(i)).getSWeasting();
			diffeasting = Math.abs(diffeasting);
			double diffnorthing = this.getNEnorthing()-((DemMetadata)listDems.get(i)).getSWnorthing();
			diffnorthing = Math.abs(diffnorthing);
			//if it does exist then add it
			if(diffeasting < 500 && diffnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				northeastdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** sets the reference to the Dem to the east of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setEastDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffNeasting = this.getNEeasting()-((DemMetadata)listDems.get(i)).getNWeasting();
			diffNeasting = Math.abs(diffNeasting);
			double diffNnorthing = this.getNEnorthing()-((DemMetadata)listDems.get(i)).getNWnorthing();
			diffNnorthing = Math.abs(diffNnorthing);
			double diffSeasting = this.getSEeasting()-((DemMetadata)listDems.get(i)).getSWeasting();
			diffSeasting = Math.abs(diffSeasting);
			double diffSnorthing = this.getSEnorthing()-((DemMetadata)listDems.get(i)).getSWnorthing();
			diffSnorthing = Math.abs(diffSnorthing);
			//if it does exist then add it
			if(diffNeasting < 500 && diffNnorthing < 500 && diffSeasting < 500 && diffSnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				eastdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** sets the reference to the Dem to the southeast of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setSoutheastDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffeasting = this.getSEeasting()-((DemMetadata)listDems.get(i)).getNWeasting();
			diffeasting = Math.abs(diffeasting);
			double diffnorthing = this.getSEnorthing()-((DemMetadata)listDems.get(i)).getNWnorthing();
			diffnorthing = Math.abs(diffnorthing);
			//if it does exist then add it
			if(diffeasting < 500 && diffnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				southeastdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** sets the reference to the Dem to the south of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setSouthDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffWeasting = this.getSWeasting()-((DemMetadata)listDems.get(i)).getNWeasting();
			diffWeasting = Math.abs(diffWeasting);
			double diffWnorthing = this.getSWnorthing()-((DemMetadata)listDems.get(i)).getNWnorthing();
			diffWnorthing = Math.abs(diffWnorthing);
			double diffEeasting = this.getSEeasting()-((DemMetadata)listDems.get(i)).getNEeasting();
			diffEeasting = Math.abs(diffEeasting);
			double diffEnorthing = this.getSEnorthing()-((DemMetadata)listDems.get(i)).getNEnorthing();
			diffEnorthing = Math.abs(diffEnorthing);
			//if it does exist then add it
			if(diffWeasting < 500 && diffWnorthing < 500 && diffEeasting < 500 && diffEnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				southdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** sets the reference to the Dem to the southwest of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setSouthwestDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffeasting = this.getSWeasting()-((DemMetadata)listDems.get(i)).getNEeasting();
			diffeasting = Math.abs(diffeasting);
			double diffnorthing = this.getSWnorthing()-((DemMetadata)listDems.get(i)).getNEnorthing();
			diffnorthing = Math.abs(diffnorthing);
			//if it does exist then add it
			if(diffeasting < 500 && diffnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				southwestdem = (DemMetadata)listDems.get(i);
			}
		}
	}
	/** sets the reference to the Dem to the west of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setWestDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffNeasting = this.getNWeasting()-((DemMetadata)listDems.get(i)).getNEeasting();
			diffNeasting = Math.abs(diffNeasting);
			double diffNnorthing = this.getNWnorthing()-((DemMetadata)listDems.get(i)).getNEnorthing();
			diffNnorthing = Math.abs(diffNnorthing);
			double diffSeasting = this.getSWeasting()-((DemMetadata)listDems.get(i)).getSEeasting();
			diffSeasting = Math.abs(diffSeasting);
			double diffSnorthing = this.getSWnorthing()-((DemMetadata)listDems.get(i)).getSEnorthing();
			diffSnorthing = Math.abs(diffSnorthing);
			//if it does exist then add it
			if(diffSeasting < 500 && diffSnorthing < 500 && diffNeasting < 500 && diffNnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				westdem = (DemMetadata)listDems.get(i);
			}
		}
	}
	/** sets the reference to the Dem to the northwest of this Dem given a Vector of DemMetadata objects to be put in the Grid object*/
	public void setNorthwestDem(Vector listDems) {
		for(int i = 0; i < listDems.size(); i++){
			double diffeasting = this.getNWeasting()-((DemMetadata)listDems.get(i)).getSEeasting();
			diffeasting = Math.abs(diffeasting);
			double diffnorthing = this.getNWnorthing()-((DemMetadata)listDems.get(i)).getSEnorthing();
			diffnorthing = Math.abs(diffnorthing);
			//if it does exist then add it
			if(diffeasting < 500 && diffnorthing < 500 && zone==((DemMetadata)listDems.get(i)).getZone()) {
				northwestdem = (DemMetadata)listDems.get(i);
			}
		}
	}

	/** gets the DemMetadata object north of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object north of this one*/
	public DemMetadata getNorthDem() {
		return northdem;
	}

	/** gets the DemMetadata object northeast of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object northeast of this one*/
	public DemMetadata getNortheastDem() {
		return northeastdem;
	}

	/** gets the DemMetadata object east of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object east of this one*/
	public DemMetadata getEastDem() {
		return eastdem;
	}

	/** gets the DemMetadata object southeast of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object southeast of this one*/
	public DemMetadata getSoutheastDem() {
		return southeastdem;
	}

	/** gets the DemMetadata object south of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object south of this one*/
	public DemMetadata getSouthDem() {
		return southdem;
	}

	/** gets the DemMetadata object southwest of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object southwest of this one*/
	public DemMetadata getSouthwestDem() {
		return southwestdem;
	}

	/** gets the DemMetadata object west of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object west of this one*/
	public DemMetadata getWestDem() {
		return westdem;
	}

	/** gets the DemMetadata object northwest of this one.
	 * returns null if it has not been set yet, or if
	 * there isn't a DemMetadata object northwest of this one*/
	public DemMetadata getNorthwestDem() {
		return northwestdem;
	}


	// accessor functions
	/**primary key */
	public String getDemId() {
		return dem_id;
	}

	/**The UTM zone */
	public int getZone() {
		return zone;
	}

	/**The number of profiles in the original DEM - think x (the max val of x in UTM coords) */
	public int getColumns_x() {
		return columns_x;
	}

	/**max_profile_length is not part of the dem specification, but is important
	* in terms imaging an entire dem. Think max "y" or column size value - note that colums are
	* normally different sizes because 1:24000 DEM data files use UTM, but the paper maps
	* are squared on lat/long, so there the column (profile) lengths adjust for the resulting
	* skew by having different sized profiles. This field counts the number of elevations
	* in the largest profile.*/
	public int getMaxProfileLength_y() {
		return max_profile_length_y;
	}

	//# corners

	/**Describes the corners of the area the DEM describes*/
	public double getSWeasting() {
		return sw_easting;
	}

	/**Describes the corners of the area the DEM describes */
	public double getSWnorthing() {
		return sw_northing;
	}

	/**Describes the corners of the area the DEM describes */
	public double getNWeasting() {
		return nw_easting;
	}

	/**Describes the corners of the area the DEM describes */
	public double getNWnorthing() {
		return nw_northing;
	}

	/**Describes the corners of the area the DEM describes */
	public double getNEeasting() {
		return ne_easting;
	}

	/**Describes the corners of the area the DEM describes */
	public double getNEnorthing() {
		return ne_northing;
	}

	/**Describes the corners of the area the DEM describes */
	public double getSEeasting() {
		return se_easting;
	}

	/**Describes the corners of the area the DEM describes */
	public double getSEnorthing() {
		return se_northing;
	}

	// misc
	/**date_added is not part of the dem specification
	* it holds the date that the DEM was placed in this database. */
	public java.sql.Date getDateAdded() {
		return date_added;
	}

	/**	The local DEM file name.
	* (Path to local DEM file - Which must be renamed for the geographic name taken from
	* the first part of the TYPE A record number 1, (first 40 chars), spaces replaced with underscores.
	* All files must have a .dem extension.
	* note that these files may use *different* horizontal and vertical datums
	* because all points in C5UTM are forcefully converted to a particular (configurable) datum.*/
	public String getLocalDemFile() {
		return local_dem_file;
	}

	/**The original file information from the origin DEM, TYPE A record number 1*/
	public String getFileInfo() {
		return file_info;
	}

	/**<p>This attribute indicates the status of statistical metadata that has been generated for
	* the areas in this dem, and stored in the tables UTM_1K_GRID_DATA and UTM_POINT_STATS,
	* See the sql for those tables. 0 indicates that the area needs
	* full processing. 1 indicates that the area was processed, but there were void
	* areas near the edges (probably because points from adjoining dems were not present in
	* the database.) 2 indicates that every point has been processed.</p>
	* <p>The UTM_1K_GRID_DATA table simply references data records in the UTM_POINT_STATS
	* table. This allows access to data representing a 1 kilometer grid overlaying the landscape
	* at even multiples of 1000 meters in the grid system. Note that this is the same as the
	* UTM grid overlaid on typical USGS maps.</p>
	@see com.c5corp.c5utm.util.Build1KGridStats
	*/
	public int get1kGridStatus() {
		return oneKgrid;
	}

	/**<p>This attribute indicates the status of statistical metadata that has been generated for
	* the areas in this dem, and stored in the tables UTM_1K_GRID_DATA and UTM_POINT_STATS,
	* See the sql for those tables. 0 indicates that the area needs
	* full processing. 1 indicates that the area was processed, but there were void
	* areas near the edges (probably because points from adjoining dems were not present in
	* the database.) 2 indicates that every point has been processed.</p>
	* <p>The UTM_POINT_STATS records represent statistics about the 1K square surrounding
	* each point in the database. This data can take an incredible amount of time to calculate</p>
	*/
	public int getPointStatStatus() {
		return point_stats;
	}

	/** gets the Average Elevation for the entire dem*/
	public double getAverageElevation(){
		return average_elevation;
	}

	/**sets boolean that indicates that the references to surrounding DemMetadata objects have been made*/
	public void setSurroundingDemsCalculated(boolean x){
		surroundingDemsCalculated = x;
	}

	/**returns boolean that indicates whether or not the surround DemMetadata object references have been calculated*/
	public boolean getSurroundingDemsCalculated(){
		return surroundingDemsCalculated;
	}

	/**returns grid X location, only meaningful after grid location has been calculated by the Grid constructor.
	 * Otherwise it returns min integer value
	 */
	public int getX() {
	   return gridX;
	}

	/**returns grid Y location, only meaningful after grid location has been calculated by the Grid constructor.
	 * Otherwise it returns min integer value
	 */
	public int getY() {
	   return gridY;
	}

	/**modifier function for Grid object to use.
	 * sets both x and y*/
	public void setGridLoc( int x, int y) {
		this.setX(x);
		this.setY(y);
	}

	/**modifier function for Grid object to use.
	 * sets x*/
	public void setX(int x) {
	   gridX = x;
	}

	/**modifier function for Grid object to use.
	 * sets y*/
	public void setY(int y) {
	   gridY = y;
	}

	/** Returns a String representation of the object */
	public String toString() {
		return	"DEM_METADATA_id = " + dem_id + "\n" +
			"zone = " +  zone + "\n" +
			"columns_x = " + columns_x + "\n" +
			"max_profile_length_y = " + max_profile_length_y + "\n" +
			"sw_easting = " + sw_easting + "\n" +
			"sw_northing = " + sw_northing + "\n" +
			"nw_easting = " + nw_easting + "\n" +
			"nw_northing = " + nw_northing + "\n" +
			"ne_easting = " + ne_easting + "\n" +
			"ne_northing = " + ne_northing + "\n" +
			"se_easting = " + se_easting + "\n" +
			"se_northing = " + se_northing + "\n" +
			"date_added = " + date_added + "\n" +
			"local_dem_file = " + local_dem_file + "\n" +
			"file_info = " + file_info + "\n" +
			"grid_stats status = " + oneKgrid + "\n" +
			"point_stats status = " + point_stats + "\n";
	}
}

