package com.c5corp.c5utm;

/** <code>public class C5UTMconfs</code>
* C5UTMconfs.java - part of the C5 Landscape database (otherwise known as the C5UTM database).
* C5UTMconfs.java reads C5UTM preferences from two files:
* conf/c5utm.conf
* - datum and units for the data in the db
* /conf/database.conf
* The location of these files can be specified using the
* constructor taking a single String (path) argument.
* - jdbc stuff and db passwords
* The results are available via pubic accessor functions
* This class also validates the configuration to the degree it can.
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import java.io.*;
import java.sql.*;
import com.c5corp.c5dem.C5DemConstants;
import java.io.File;

public class C5UTMconfs {
	// declarations
	/**vertical datum*/
	String vdatum = null;
	/**horizontal datum*/
	String hdatum = null;
	/**vertical units*/
	String vunits = null;
	/**horizontal units*/
	String hunits = null;
	/**database url*/
	String db_url = null;
	/**database driver*/
	String db_driver = null;
	private String access_password = null;
	private String update_password = null;

	// default values for various important directories coded here
	// can override in the conf/c5utm.conf file
	// by definition, this is not 100% pure java.
	/**path to the corpscon directory*/
	static String corpscon_dir = "CORPSCON";
	/**path to the dem directory*/
	static String dem_dir = "DEM";
	/**path to the receipts directory*/
	static String receipts_dir = "RECEIPTS";
	/**path to the new dems directory*/
	static String newdems_dir = "NEW_DEMS";
	/**path to the public html directory*/
	static String public_html = "public_html" + File.separator;
	/**local path to the utm image directory*/
	static String utm_img_dir = public_html + "utm_img";
	/**local path to the image cache directory*/
	static String image_cache = public_html + "img_cache";

	// the base url of the web installation - if configured in c5utm.conf
	// better urls can be reported in the receipt files in the RECEIPTS
	// directory.
	/**the base utl to the web installation*/
	static String base_url = "";

	private boolean units_config = false;
	private boolean config_validation_ok = false;
	private boolean database_working = true; // true until proven false.
	private String messages = "C5UTMconf messages (all ok if none):\n";

	/** This constructor opens and parses the files
	* and loads the private vars*/
	// default
	public C5UTMconfs() {
		this("conf" + File.separatorChar);
	}

	/** This constructor opens and parses the files
	* and loads the private vars.
	* @param path representing the path to the conf directory and the base instalation
	* must be called with a path to the directory containing the conf files*/
	public C5UTMconfs(String path) {

		BufferedReader config = null;
		String str = null;
		String field = null;
		boolean url = false;
		boolean driver = false;
		boolean access = false;
		boolean update = false;
		boolean vdat = false;
		boolean hdat = false;
		boolean vuni = false;
		boolean huni = false;
		boolean db_driver_ok = true; // true until proven false
		boolean c5utmconf_open = true; // true until proven false
		boolean add2c5utmconf_open = true; // true until proven false

		// trim whitespace
		path = path.trim();
		// if the path does not end in File.pathSeparator add it
		if (path.lastIndexOf(File.separatorChar) != path.length()-1) {
			path = path + File.separatorChar;
		}

		String c5utmconf_file = path + "c5utm.conf";
		String database_conf = path + "database.conf";

		// read the c5utm.conf file and get the units for this install
		try {
			config = new BufferedReader(new FileReader(c5utmconf_file));
		} catch (FileNotFoundException e) {
			messages += "Can't open: " + c5utmconf_file + "\n";
			c5utmconf_open = false;
		}

		if (c5utmconf_open) {
			try {
				READING: while (config.ready()) {
					str = config.readLine();
					// skip comments
					if (str.charAt(0) == '#') {
						continue;
					}
					field = str.substring(0, str.indexOf(','));
					if (field.equals("vertical_datum")) {
						vdatum = str.substring(str.indexOf(',') + 1, str.length());
						for (int i=0; i < C5DemConstants.vdatum.length; i++) {
							if (C5DemConstants.vdatum[i].equals(vdatum)) {
								vdat = true;
								continue READING;
							}
						}
						messages += "vertical_datum not configured correctly in " + c5utmconf_file + "\n";
					} else if (field.equals("horizontal_datum")) {
						hdatum = str.substring(str.indexOf(',') + 1, str.length());
						for (int i=0; i < C5DemConstants.hdatum.length; i++) {
							if (C5DemConstants.hdatum[i].equals(hdatum)) {
								hdat = true;
								continue READING;
							}

						}
						messages += "horizontal_datum not configured correctly in " + c5utmconf_file + "\n";
					} else if (field.equals("vertical_units")) {
						vunits = str.substring(str.indexOf(',') + 1, str.length());
						for (int i=0; i < C5DemConstants.units.length; i++) {
							if (C5DemConstants.units[i].equals(vunits)) {
								vuni = true;
								continue READING;
							}
						}
						messages += "vertical_units not configured correctly in " + c5utmconf_file + "\n";
					} else if (field.equals("horizontal_units")) {
						hunits = str.substring(str.indexOf(',') + 1, str.length());
						for (int i=0; i < C5DemConstants.units.length; i++) {
							if (C5DemConstants.units[i].equals(hunits)) {
								huni = true;
								continue READING;
							}
						}
						messages += "horizontal_units not configured correctly in " + c5utmconf_file + "\n";
					} else if (field.equals("corpscon_dir")) {
						corpscon_dir = str.substring(str.indexOf(',') + 1, str.length());
					} else if (field.equals("dem_dir")) {
						dem_dir = str.substring(str.indexOf(',') + 1, str.length());
					} else if (field.equals("receipts_dir")) {
						receipts_dir = str.substring(str.indexOf(',') + 1, str.length());
					} else if (field.equals("newdems_dir")) {
						newdems_dir = str.substring(str.indexOf(',') + 1, str.length());
					} else if (field.equals("base_url")) {
						base_url = str.substring(str.indexOf(',') + 1, str.length());
					} else if (field.equals("public_html")) {
						public_html = str.substring(str.indexOf(',') + 1, str.length());
					}else if (field.equals("utm_img_dir")) {
						utm_img_dir = str.substring(str.indexOf(',') + 1, str.length());
					}else if (field.equals("image_cache")) {
						image_cache = str.substring(str.indexOf(',') + 1, str.length());
					}
				}
			} catch (IOException e) {
				System.err.println(e);
			}
		}


		if (vdat && hdat && vuni && huni) {
			units_config = true;
		}

		config = null;

		// open the config file - ../conf/database.conf
		// a possible exception here is that the user does not have
		// permission to read this file - (it contains DB passwords)
		try {
			config = new BufferedReader(new FileReader(database_conf));
		} catch (FileNotFoundException e) {
			messages += "Can't open: " + database_conf + "\n";
			add2c5utmconf_open = false;
		}

		// make sure the correct info is in the database.conf file
		if (add2c5utmconf_open) {
			try {
				while (config.ready()) {
					str = config.readLine();
					// skip comments
					if (str.charAt(0) == '#') {
						continue;
					}
					field = str.substring(0, str.indexOf(','));
					if (field.equals("jdbc_url")) {
						db_url = str.substring(str.indexOf(',') + 1, str.length());
						url = true;
					} else if (field.equals("jdbc_driver")) {
						db_driver = str.substring(str.indexOf(',') + 1, str.length());
						driver = true;
					} else if (field.equals("access_password")) {
						access_password = str.substring(str.indexOf(',') + 1, str.length());
						access = true;
					} else if (field.equals("update_password")) {
						update_password = str.substring(str.indexOf(',') + 1, str.length());
						update = true;
					}
				}
			} catch (IOException e) {
				System.err.println(e);
			}

			if (!(url && driver && access && update)) { // if conf file is not complete
				messages += "config warning: db_url, db_driver, access_password, and update password\n";
				messages += "must be configured in " + database_conf + "\n";
			} else { // if it is, set this flag
				config_validation_ok = true;
			}
		}

		// check to see if the database is installed and present -set database validation
		if (config_validation_ok) {
			try {
				// The newInstance() call is a work around for some
                    		// broken Java implementations
				Class.forName(db_driver).newInstance();
			} catch (Exception e) {
				messages += "Failed to make DB connection using driver: "
					+ db_driver + " , is it installed correctly?\n";
				db_driver_ok = false;
				database_working = false;
			}

			if (db_driver_ok) {
				try {
					Connection connection=DriverManager.getConnection(db_url, "C5UTM_access", access_password);
					connection.close();     // close db connection
				} catch (Exception e) {
					messages += "C5UTM_access password on " + db_url + " failed.\n";
					database_working = false;
				}

				try {
					Connection connection=DriverManager.getConnection(db_url, "C5UTM_update", update_password);
					connection.close();     // close db connection
				} catch (Exception e) {
					messages += "C5UTM_update password on " + db_url + " failed.\n";
					database_working = false;
				}
			}
		}
	}

	// the public accessor functions allow access to the configurations
	/** Returns a java.lang.String containing the configured vertical datum
	* for this C5UTM database installation.
	*/
	public String getVerticalDatum() {
		return vdatum;
	}

	/** Returns a java.lang.String containing the configured horizontal datum
	* for this C5UTM database installation.
	*/
	public String getHorizontalDatum() {
		return  hdatum;
	}

	/** Returns a java.lang.String containing the configured vertical units
	* for this C5UTM database installation.
	*/
	public String getVerticalUnits() {
		return vunits;
	}

	/** Returns a java.lang.String containing the configured horizontal units
	* for this C5UTM database installation.
	*/
	public String getHorizontalUnits() {
		return hunits;
	}

	/** Returns a java.lang.String containing the configured CORPSCON directory
	* for this C5UTM database installation.
	*/
	public String getCorpsconDir() {
		return corpscon_dir;
	}

	/** Returns a java.lang.String containing the configured DEM directory,
	* (the location of dems already processed into the database),
	* for this C5UTM database installation.
	*/
	public String getDemDir() {
		return dem_dir;
	}

	/** Returns a java.lang.String containing the configured RECEIPTS directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public String getReceiptsDir() {
		return receipts_dir;
	}

	/** Returns a java.lang.String containing the configured NEW_DEMS directory,
	* (the location of dems waiting to be processed into the database),
	* for this C5UTM database installation.
	*/
	public String getNewDemsDir() {
		return newdems_dir;
	}

	/** Returns a java.lang.String containing the configured web directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public String getPublicHtml() {
		return public_html;
	}

	/**deprecated Returns a java.lang.String containing the configured web image directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public String getWebImageDir() {
		return utm_img_dir;
	}
	/** Returns a java.lang.String containing the configured web image directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public String getUTMImageDir() {
		return utm_img_dir;
	}

	/** Returns a java.lang.String containing the configured web image cache directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public String getImageCacheDir() {
		return image_cache;
	}

	/** Returns a java.io.File containing the configured CORPSCON directory
	* for this C5UTM database installation.
	*/
	public File getCorpsconDirFile() {
		File file =  new File(corpscon_dir);
		if (!file.isDirectory()) {
			System.err.println("Configuration error: the file configured as the CORPSCON directory is not a directory.");
		}
		return file;
	}

	/** Returns a java.io.File containing the configured DEM directory,
	* (the location of dems already processed into the database),
	* for this C5UTM database installation.
	*/
	public File getDemDirFile() {
		File file =  new File(dem_dir);
		if (!file.isDirectory()) {
			System.err.println("Configuration error: the file configured as the DEM directory is not a directory.");
		}
		return file;
	}

	/** Returns a java.io.File containing the configured RECEIPTS directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public File getReceiptsDirFile() {
		File file =  new File(receipts_dir);
		if (!file.isDirectory()) {
			System.err.println("Configuration error: the file configured as the RECEIPTS directory is not a directory.");
		}
		return file;
	}

	/** Returns a java.io.File containing the configured NEW_DEMS directory,
	* (the location of dems waiting to be processed into the database),
	* for this C5UTM database installation.
	*/
	public File getNewDemsDirFile() {
		File file =  new File(newdems_dir);
		if (!file.isDirectory()) {
			System.err.println("Configuration error: the file configured as the NEW_DEMS directory is not a directory.");
		}
		return file;
	}

	/** Returns a java.io.File containing the configured web image directory
	* for this C5UTM database installation. (Receipts are database management history info.)
	*/
	public File getUTMImageDirFile() {
		File file =  new File(utm_img_dir);
		if (!file.isDirectory()) {
			System.err.println("Configuration error: the file configured as the utm_img directory file is not a directory.");
		}
		return file;
	}

	/** Returns a java.lang.String containing the configured base url
	* for this C5UTM database installation.
	*/
	public String getBaseUrl() {
		return base_url;
	}

	/** Returns a java.lang.String containing the configured database url
	* for this C5UTM database installation.
	*/
	public String getDbUrl() {
		return db_url;
	}

	/** Returns a java.lang.String containing the configured database driver
	* for this C5UTM database installation.
	*/
	public String getDriver() {
		return db_driver;
	}

	/** Returns a java.lang.String containing the configured access password
	* for this C5UTM database installation.
	*/
	public String getAccessPassword() {
		return access_password;
	}

	/** Returns a java.lang.String containing the configured update password
	* for this C5UTM database installation.
	*/
	public String getUpdatePassword() {
		return update_password;
	}

	// If the configuration file validates and the DB is working, this will be true.
	//
	/** Returns a boolean, true if the configuration file validates and the DB is working.
	*/
	public boolean databaseWorking() {
		return config_validation_ok && database_working;
	}

	/** Returns a boolean, true if the database units are configured.
	*/
	public boolean databaseUnitsConfigured() {
		return units_config;
	}

	/** Returns a java.lang.String, containing any diagnostic messages related to datbase configuration and validation.
	*/
	public String getMessages() {
		return messages;
	}

}
