package com.c5corp.c5utm;

import java.io.*;
import java.sql.*;
import java.util.*;
import com.c5corp.c5dem.UtmCoordinatePairElev;
import com.c5corp.c5dem.DemTable;

/**
* <p>C5UTM contains static methods for accessing the data in a C5 Landscape database installation.</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0.3
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class C5UTM {

	/**
	* getPoint takes the UTM zone, easting, and northing values of a target site.
	* (And a connection object.)
	* It returns a Point object describing the point, (factory method approach)
	* Example: C5UTM.getPoint(zone, easting, northing, connection);
	* This version of the function is designed to "snap to grid" finding the closest
	* point that it can. The object returned will contain the actual UTM easting and
	* northing that was found in the database. Returns null if point not found.</p>
	* @param zone the UTM zone
	* @param easting the UTM easting
	* @param northing the UTM northing
	* @param connection the database connection object
	* @see DbHelper
	* @see Point
	* @return A Point object representing the result
	*/
	public static Point getPoint(int zone, int easting, int northing, Connection connection) {

		int easthigh=easting+15;
		int northhigh=northing+15;
		int eastlow=easting-15;
		int northlow=northing-15;
		int adjusted_easting=0;
		int adjusted_northing=0;
		int elevation=0;
		String DEM_METADATA_id="";
		String id="";

		try {
			// Connection connection = null; // db conncetion
			Statement statement = null; // sql to send to db
			ResultSet resultset; // results from query

			// prepare the statement
			statement = connection.createStatement();

			// execute the db query
			resultset = statement.executeQuery("SELECT UTM_COORD.easting, UTM_COORD.northing, " +
			"UTM_COORD.elevation, UTM_COORD.DEM_METADATA_id, UTM_COORD_id " +
			"FROM UTM_COORD, DEM_METADATA " +
			"WHERE (DEM_METADATA.zone=" + zone +
			" && DEM_METADATA.DEM_METADATA_id=UTM_COORD.DEM_METADATA_id)" +
			" && (UTM_COORD.easting BETWEEN " + eastlow + " && " + easthigh + ")" +
			" && (UTM_COORD.northing BETWEEN " + northlow + " && " + northhigh + ")");

			// the resultset for this query should always be one row

			// the zone will always be the same as the input...
			// zone = zone
			if (resultset.next()) {
				adjusted_easting = resultset.getInt("easting");
				adjusted_northing = resultset.getInt("northing");
				elevation = resultset.getInt("elevation");
				DEM_METADATA_id = resultset.getString("DEM_METADATA_id");
				id = resultset.getString("UTM_COORD_id");
				try {
					statement.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			} else {
				try {
					statement.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
				return null;
			}
		} catch (SQLException e) {

			e.printStackTrace();
		}

		// wrap it up and return as a point object.
		return new Point(zone, adjusted_easting, adjusted_northing, elevation, DEM_METADATA_id, id);
	}


	/**
	* Given the database id (which is the primary key field) of the record, this method
	* returns a Point object for the point, factory method approach.
	* Example: C5UTM.getPointById(a_32_character_md5_hash_string, connection);
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static Point getPointById(String id, Connection connection) {

		int zone=0;
		int adjusted_easting=0;
		int adjusted_northing=0;
		int elevation=0;
		String DEM_METADATA_id="";
		// Connection connection = null; // db conncetion
		Statement statement = null; // sql to send to db
		ResultSet resultset; // results from query

		// get db conncetion
		// connection=DbHelper.getDbConnection();
		try {
			// prepare the statement
			statement = connection.createStatement();

			// execute the db query
			resultset = statement.executeQuery("SELECT DEM_METADATA.zone UTM_COORD.easting, UTM_COORD.northing, " +
			"UTM_COORD.elevation, UTM_COORD.DEM_METADATA_id " +
			"FROM UTM_COORD, DEM_METADATA " +
			"WHERE UTM_COORD_id=" + id + " && DEM_METADATA.DEM_METADATA_id=UTM_COORD.DEM_METADATA_id");

			// the resultset for this query should always be one row

			if (resultset.next()) {
				zone = resultset.getInt("zone");
				adjusted_easting = resultset.getInt("easting");
				adjusted_northing = resultset.getInt("northing");
				elevation = resultset.getInt("elevation");
				DEM_METADATA_id = resultset.getString("DEM_METADATA_id");
			} else {
				statement.close();
				return null;
			}

			statement.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		// wrap it up and return as a point object.
		return new Point(zone, adjusted_easting, adjusted_northing, elevation, DEM_METADATA_id, id);
	}

	/**
	* This method is similar to the getPoints method in this class, (which returns a more powerful Points object)
	* in that it has the same method
	* signature and performs a similar function, but has some notable differences. As of version 1.2,
	* it is assumed that the DEM files are maintained in addition to the (same) data in the rdbms.
	* DEMs are not very expensive to store, and are faster to process in many circumstances. Although living
	* in the C5UTM
	* class, this method will actually <i>go to the files first</i> for data if the data asked for is <i>contiguous
	* in the file</i>. If not, it will go to the rdmbs for the data. Note that it returns a (two dimensional)
	* array of a com.c5corp.c5dem type (UtmCoordinatePairElev),
	* and not a com.c5corp.c5utm type. The raw dems don't have all of the data that the database does, (for
	* example, an internal array of Point objects that have an id and dem_id field, or various statistical
	* calculations and fields.) So given that this method <i>may</i> go to the file instead of the db, it
	* returns the lowest common denominator type of the two potential return types.
	* @param zone the UTM zone
	* @param easting the UTM easting
	* @param northing the UTM northing
	* @param easting_meters meters east of easting-northing requested
	* @param  northing_meters meters north of easting-northing requested
	* @param connection the database connection object
	* @return an array representing the requested area, or null if the data could not be found.
	* @since 1.0.3
	* @see #getPoints
	* @see com.c5corp.c5dem.UtmCoordinatePairElev
	*/
	public static UtmCoordinatePairElev[][] getCoordinatePairElevArray(int zone, int easting, int northing,
				int easting_meters, int northing_meters, Connection connection) {

		UtmCoordinatePairElev[][] data = null;

		// examine DEM_METADATA and see if we can find all of the data in one DEM.
		DemMetadata demMetadata = findDem(zone, easting, northing, easting_meters, northing_meters, connection);

		if (demMetadata == null) {// if not one found
			// use getPoints to get the data
			Points points = getPoints(zone, easting, northing, easting_meters, northing_meters, connection);
			// convert the Points it into a UtmCoordinatePairElev[][]
			if (points != null) {
				data = points.getUtmCoordinatePairElevArray();
			}
		} else { // If we found a fully enclosing dem, use it
			// get the UtmCoordinatePairElev[][] from the file
			C5UTMconfs confs = new C5UTMconfs();
			DemTable dem = new DemTable(confs.getDemDir() + File.separator + demMetadata.getLocalDemFile());
			// convert to the vertical units configured in the database if necessary
			if (confs.getVerticalUnits().equals("feet")) {
				dem.convertToVerticalFeet();
			} else if (confs.getVerticalUnits().equals("meters")) {
				dem.convertToVerticalMeters();
			}
			data = dem.getPoints(easting, northing, easting_meters, northing_meters);
		}

		// return it
		return data;
	}

	/**
	* Given an area represented by the methods parameters, a DemMetadata object
	* representing a Dem that encloses the given area. Returns null if it can not
	* determine that the given area is entirely enclosed by one Dem, indicating
	* that (most likely) the area contains points from one or more Dems.
	* @param zone the UTM zone
	* @param easting the UTM easting
	* @param northing the UTM northing
	* @param easting_meters meters east of easting-northing requested
	* @param  northing_meters meters north of easting-northing requested
	* @param connection the database connection object
	* @return a Dem
	* @since 1.0.3
	*/
	public static DemMetadata findDem(int zone, int easting, int northing,
				int easting_meters, int northing_meters, Connection connection) {
		DemMetadata demMeta = null;

		try {

			Statement select = connection.createStatement();
			ResultSet result = select.executeQuery
			("SELECT DEM_METADATA_id, zone, columns_x, max_profile_length_y, \n" +
				"sw_easting, sw_northing, nw_easting, nw_northing, \n" +
				"ne_easting, ne_northing, se_easting, se_northing, \n" +
				"date_added, local_dem_file, file_info, \n" +
				"1K_grid, point_stats, average_elevation \n" +
				"FROM DEM_METADATA where \n" +
				"zone = " + zone + " && \n" +
				"(" + easting + " > sw_easting && " + easting + " > nw_easting) && \n" +
				"(" + (easting + easting_meters) + " < se_easting && " + (easting + easting_meters)  + " < ne_easting) && \n" +
				"(" + northing + " > sw_northing && " + northing + " > se_northing) && \n" +
				"(" + (northing + northing_meters) +  " < nw_northing && " + (northing + northing_meters) + " < ne_northing)\n"
			);

			while (result.next()) { // should only be one or zero results
				demMeta = new DemMetadata(
						result.getString("DEM_METADATA_id"),
						result.getInt("zone"),
						result.getInt("columns_x"),
						result.getInt("max_profile_length_y"),
						result.getDouble("sw_easting"),
						result.getDouble("sw_northing"),
						result.getDouble("nw_easting"),
						result.getDouble("nw_northing"),
						result.getDouble("ne_easting"),
						result.getDouble("ne_northing"),
						result.getDouble("se_easting"),
						result.getDouble("se_northing"),
						result.getDate("date_added"),
						result.getString("local_dem_file"),
						result.getString("file_info"),
						result.getInt("1K_grid"),
						result.getInt("point_stats"),
						result.getDouble("average_elevation")
				);
			}
			select.close();
			result.close();

		} catch (SQLException e) {
			System.err.println("Exception when checking for pre-existing entry.");
			e.printStackTrace();
		}

		return demMeta;
	}


	/**
	* returns a Points object - factory method approach
	* Note that some or all requested Points might not be in the database.
	* the returned Points object will contain only as many as it could find
	* meeting the criteria.
	* Example: getPoints(zone, easting, northing, east_meters, north_meters, connection);<br>
	* not implemented: Can't yet get interpolate points from neighboring UTM zones...</p>
	* @param zone the UTM zone
	* @param easting the UTM easting
	* @param northing the UTM northing
	* @param easting_meters meters east of easting-northing requested
	* @param  northing_meters meters north of easting-northing requested
	* @param connection the database connection object
	* @return the Points object representing the area requested
	* @see DbHelper
	*/
	@SuppressWarnings("unchecked")
	public static Points getPoints(int zone, int easting, int northing,
				int easting_meters, int northing_meters, Connection connection) {

		// use getPoint to snap to nearest point
		Point sw_corner = getPoint(zone, easting, northing, connection);
		// if the SW corner is null we return null
		if (sw_corner == null) {
			return null;
		}

		// some more declarations
		int east_bounds = easting+easting_meters;
		int north_bounds = northing+northing_meters;
		int eastmost_easting=-1;
		int northmost_northing=-1;
		int ewidth=0;	// these for the x and y values
		int eheight=0;
		int ewatch=0;	// used to watch for a new col starting
		int lowest=-10000;
		int highest=100000;
		int largest_col=0;
		Point[][] points = null;
		Vector<Vector> points_temp = null;
		int[] shape = null;
		Vector<Integer> shape_temp = null;
		boolean has_void = false;
		// Connection connection = null; // db conncetion
		Statement statement = null; // sql to send to db
		ResultSet resultset; // results from query

		// Reuse the easting and northing params - snapping to grid happens here
		easting = sw_corner.getEasting();
		northing = sw_corner.getNorthing();

		try {
			// prepare the statement
			statement = connection.createStatement();

			// execute the db query - this sql will only get points within the zone
			resultset = statement.executeQuery("SELECT UTM_COORD.easting, UTM_COORD.northing, " +
			"UTM_COORD.elevation, DEM_METADATA.DEM_METADATA_id, UTM_COORD_id " +
			"FROM UTM_COORD, DEM_METADATA " +
			"WHERE ((DEM_METADATA.zone=" + zone + " && DEM_METADATA.DEM_METADATA_id=UTM_COORD.DEM_METADATA_id)" +
			"	&& (UTM_COORD.easting BETWEEN " + easting + " && " + east_bounds + ")" +
			"	&& (UTM_COORD.northing BETWEEN " + northing + " && " + north_bounds + "))" +
			"ORDER BY UTM_COORD.easting, UTM_COORD.northing");

			// make point_temp and shape_temp - using Vectors to fill the data structures,
			// but will convert to arrays later.
			points_temp = new Vector<Vector>();
			shape_temp = new Vector<Integer>();

			// parse the results into a 2d array (array of refs to arrays...)
			if (resultset.next()) {	// must get initial row to watch for easting change - priming read
				// place Point 0,0 into the 2d Vector point_temp
				Vector<Point> newvec = new Vector<Point>();
				newvec.add(sw_corner);
				points_temp.add(newvec);

				ewatch = sw_corner.getEasting(); // checking this value (easting) for changes
								// (the data from the db is 1D, need to watch to extract 2D
				eheight++;
				lowest = sw_corner.getElevation(); 	// the first point - priming read
				highest = sw_corner.getElevation();
			}


			while (resultset.next()) {
				easting = resultset.getInt("easting");
				northing = resultset.getInt("northing");
				int elevation = resultset.getInt("elevation");
				String DEM_METADATA_id = resultset.getString("DEM_METADATA_id");
				String id = resultset.getString("UTM_COORD_id");

				// watch for the easting to change by more than +- 1
				// and start new row in 2d array

				if ((ewatch > easting+1) || (ewatch < easting-1) ) {
					shape_temp.add(new Integer(eheight));
					if (largest_col < eheight) largest_col=eheight;
					ewidth++;
					eheight = 0;
					Vector newvec = new Vector();
					points_temp.add(newvec);
				}

				// find the highest actual northing and easting values found
				if (eastmost_easting < easting) eastmost_easting = easting;
				if (northmost_northing < northing) northmost_northing = northing;

				// see note above sql - this function only gets points from within $zone
				Vector<Point> tempVec = points_temp.get(ewidth);
				tempVec.add(new Point(zone, easting, northing, elevation, DEM_METADATA_id, id));

				eheight++;
				ewatch = easting;

				// capture the current lowest
				if (elevation < lowest) lowest = elevation;

				// capture the current highest
				if (elevation > highest) highest = elevation;
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}

		ewidth++;

		shape_temp.add(new Integer(eheight));

		// convert vecs to arrays
		//points = new Point[ewidth][largest_col];
		points = new Point[ewidth][];
		shape = new int[ewidth];

		for (int i = 0; i < ewidth; i++) {
			Integer the_int = (Integer)shape_temp.get(i);
			shape[i] = the_int.intValue();
			points[i] = new Point[((Vector)points_temp.get(i)).size()];
			for (int j = 0; j < ((Vector)points_temp.get(i)).size(); j++) {
				points[i][j] = (Point)((Vector)points_temp.get(i)).get(j);
			}
		}

		// determine if there are void areas
		// A 1K area should be a 34x34 array
		// (ewidth and largest col should be 34)
		int point_count = 0;
		int void_count = 0;
		for (int i = 0; i < 34; i++) {
			for (int j = 0; j < 34; j++) {
				point_count++;
				if ((i > ewidth) || (j > largest_col)) {
					has_void = true;
					void_count++;
				}
			}
		}

		double percent_void = (void_count/(double)point_count)*100;

		try {
			statement.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return new Points(lowest, highest, largest_col, points, shape, has_void, percent_void);
	}

	/** <code>public static Vector getDemMetadataSearch(String query)</code>
	* Strings are searched in the DEM_METADATA.file_info field (case insensitve), and a Vector
	* of DemMetadata objects is returned that match the search; possibly null,
	* if no matches are found. The search string can contain double quoted literals, otherwise
	* the arguments are split on white space and search withed 'and' logic.
	* @param queryStr the query string
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static Vector getDemMetadataSearch(String queryStr, Connection connection) {
		Vector<DemMetadata> results = new Vector<DemMetadata>();
		Vector<String> query = new Vector<String>(); // the queryStr split at whitespace
		String searchStr = ""; // initialize to empty string
		//Connection connection = null; // db conncetion
		Statement statement = null; // sql to send to db
		ResultSet resultset; // results from query

		// parse the query string
		queryStr = queryStr.trim();
		StringTokenizer tk = new StringTokenizer(queryStr);
		while (tk.hasMoreTokens()) {
			String word = tk.nextToken();
			if (!word.equals("")) { // eat multiple spaces
				query.add(word);
			}
		}

		// return null if no search string
		if (query.size() == 0) return null;

		//form the string to append to the sql
		for (int i = 0; i < query.size()-1; i++) {
			searchStr = "file_info LIKE '%" + query.elementAt(i) + "%' && ";
		}
		searchStr = searchStr + "file_info LIKE '%" + query.elementAt(query.size()-1) + "%'";

		try {

			// get db conncetion
			//connection=DbHelper.getDbConnection();

			// prepare the statement
			statement = connection.createStatement();

			// execute the db query
			resultset = statement.executeQuery(
				"SELECT DEM_METADATA_id, zone, columns_x, max_profile_length_y, sw_easting," +
				"sw_northing, nw_easting, nw_northing, ne_easting, ne_northing, " +
				"se_easting, se_northing, " +
				" date_added, local_dem_file, file_info, " +
				"1K_grid, point_stats FROM DEM_METADATA where " + searchStr
			);

			while (resultset.next()) {
				results.add(new DemMetadata(
					resultset.getString("DEM_METADATA_id"),
					resultset.getInt("zone"),
					resultset.getInt("columns_x"),
					resultset.getInt("max_profile_length_y"),
					resultset.getDouble("sw_easting"),
					resultset.getDouble("sw_northing"),
					resultset.getDouble("nw_easting"),
					resultset.getDouble("nw_northing"),
					resultset.getDouble("ne_easting"),
					resultset.getDouble("ne_northing"),
					resultset.getDouble("se_easting"),
					resultset.getDouble("se_northing"),
					resultset.getDate("date_added"),
					resultset.getString("local_dem_file"),
					resultset.getString("file_info"),
					resultset.getInt("1K_grid"),
					resultset.getInt("point_stats"),
					resultset.getDouble("average_elevation")
				));
			}
		} catch (SQLException e) {
			//results = null; // Attempt to abuse the Vector - this is good, Generics found it.
			//results = new Vector<DemMetadata>();
			//results.add(e.toString());
			e.printStackTrace();
		}

		try {
			statement.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return results;
	}



	/** gets all the Dem meta data objects from the database*/
	public static Vector<DemMetadata> getAllDemMetadata(Connection connection) {
		Vector<DemMetadata> results = new Vector<DemMetadata>();


		//Connection connection = null; // db conncetion
		Statement statement = null; // sql to send to db
		ResultSet resultset; // results from query

		try {

			// get db conncetion
			//connection=DbHelper.getDbConnection();

			// prepare the statement
			if(connection != null)
			{
				statement = connection.createStatement();
			}
			// execute the db query
			resultset = statement.executeQuery(
				"SELECT DEM_METADATA_id, zone, columns_x, max_profile_length_y, sw_easting," +
				"sw_northing, nw_easting, nw_northing, ne_easting, ne_northing, " +
				"se_easting, se_northing, " +
				" date_added, local_dem_file, file_info, " +
				"1K_grid, point_stats FROM DEM_METADATA ORDER BY nw_northing DESC, nw_easting ASC"	);

			while (resultset.next()) {
				results.add(new DemMetadata(
					resultset.getString("DEM_METADATA_id"),
					resultset.getInt("zone"),
					resultset.getInt("columns_x"),
					resultset.getInt("max_profile_length_y"),
					resultset.getDouble("sw_easting"),
					resultset.getDouble("sw_northing"),
					resultset.getDouble("nw_easting"),
					resultset.getDouble("nw_northing"),
					resultset.getDouble("ne_easting"),
					resultset.getDouble("ne_northing"),
					resultset.getDouble("se_easting"),
					resultset.getDouble("se_northing"),
					resultset.getDate("date_added"),
					resultset.getString("local_dem_file"),
					resultset.getString("file_info"),
					resultset.getInt("1K_grid"),
					resultset.getInt("point_stats"),
					resultset.getDouble("average_elevation")
				));
			}
		} catch (SQLException e) {
//			results = null;
//			results = new Vector();
//			results.add(e.toString());
			e.printStackTrace();
		}

		try {
			statement.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return results;
	}

	/** <code>public static DemMetadata getDemMetadata(int id)</code>
	* Searches the DEM_METADATA.DEM_METADATA_id field for
	* a matching DEM_METADATA_id, returning either a DemMetadata object, or null if there
	* is no such DEM_METADATA_id. (Point objects contain the DEM_METADATA_id of the dem they lie in...)
	* @param id the query id
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static DemMetadata getDemMetadata(String id, Connection connection) {

		//Connection connection = null; // db conncetion
		Statement statement = null; // sql to send to db
		ResultSet resultset; // results from query
		DemMetadata result = null;

		try {

  			 // get db conncetion
			// connection=DbHelper.getDbConnection();

			// prepare the statement
			statement = connection.createStatement();

			// execute the db query
			resultset = statement.executeQuery("SELECT DEM_METADATA_id, zone, columns_x, " +
				"max_profile_length_y, sw_easting," +
				"sw_northing, nw_easting, nw_northing, ne_easting, " +
				"ne_northing, se_easting, se_northing, date_added, " +
				"local_dem_file, file_info," +
				"1K_grid, point_stats FROM DEM_METADATA where DEM_METADATA_id=\"" + id + "\"");

			if (resultset.next()) {
				result = new DemMetadata(
					resultset.getString("DEM_METADATA_id"),
					resultset.getInt("zone"),
					resultset.getInt("columns_x"),
					resultset.getInt("max_profile_length_y"),
					resultset.getDouble("sw_easting"),
					resultset.getDouble("sw_northing"),
					resultset.getDouble("nw_easting"),
					resultset.getDouble("nw_northing"),
					resultset.getDouble("ne_easting"),
					resultset.getDouble("ne_northing"),
					resultset.getDouble("se_easting"),
					resultset.getDouble("se_northing"),
					resultset.getDate("date_added"),
					resultset.getString("local_dem_file"),
					resultset.getString("file_info"),
					resultset.getInt("1K_grid"),
					resultset.getInt("point_stats"),
					resultset.getDouble("average_elevation")
				);
			}
		} catch (SQLException e) {
			result = null;

			e.printStackTrace();
		}

		try {
			statement.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return result;
	}

	/** <code>Vector findExactDemNames(String str, C5UTMconfs conf)</code>
	* Finds an exact match for a dem name from the file_info field of DEM_METADATA.
	* This method will normally return a Vector with either 0 or 1 elements. The
	* objects in the Vector are DemMetadata objects.
	* If it returns more elements, you may wish to investigate
	* that there is not a duplicate map in the database.
	* If you have a confs object available, use this version.
	* @see com.c5corp.c5utm.DemMetadata
	* @see com.c5corp.c5utm.C5UTMconfs
	* @param str the query string
	* @param confs the configuration object
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static Vector findExactDemNames(String str, C5UTMconfs confs, Connection connection) {
		Vector<DemMetadata> vec = new Vector<DemMetadata>();
		try {

			Statement select = connection.createStatement();
			ResultSet result = select.executeQuery
				("SELECT * FROM DEM_METADATA where file_info LIKE '" + str + "'");

			while (result.next()) {
				vec.add(new DemMetadata(
						result.getString("DEM_METADATA_id"),
						result.getInt("zone"),
						result.getInt("columns_x"),
						result.getInt("max_profile_length_y"),
						result.getDouble("sw_easting"),
						result.getDouble("sw_northing"),
						result.getDouble("nw_easting"),
						result.getDouble("nw_northing"),
						result.getDouble("ne_easting"),
						result.getDouble("ne_northing"),
						result.getDouble("se_easting"),
						result.getDouble("se_northing"),
						result.getDate("date_added"),
						result.getString("local_dem_file"),
						result.getString("file_info"),
						result.getInt("1K_grid"),
						result.getInt("point_stats"),
						result.getDouble("average_elevation")
					)
				);
			}
			select.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return vec;
	}

	/** <code>Vector findExactDemNames(String str)</code>
	* Finds an exact match for a dem name from the file_info field of DEM_METADATA.
	* This method will normally return a Vector with either 0 or 1 element
	* of type DemMetadata. If it returns more elements, you may wish to investigate
	* that there is not a duplicate map in the database. If you have a confs
	* object available, use <code>Vector findLikeDemNames(String str, C5UTMconfs conf)</code>
	* This overloaded version is slower because it has the overhead of making a new C5UTMconfs
	* @see com.c5corp.c5utm.DemMetadata
	* @param str the query string
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static Vector findExactDemNames(String str, Connection connection) {
		C5UTMconfs confs = new C5UTMconfs();
		return findExactDemNames(str, confs, connection);
	}

	/** <code>Vector findDemNameContains(String str, C5UTMconfs conf)</code>
	* This method will normally return a Vector with either 0 or more elements
	* of type DemMetadata. It searches the file_info field of DEM_METADATA for a
	* subsring containing the query string. If you have a confs
	* object available, use this version. It is faster not to make a new C5UTMconfs.
	* @see com.c5corp.c5utm.DemMetadata
	* @see com.c5corp.c5utm.C5UTMconfs
	* @param str the query string
	* @param confs the configuration object
	* @param connection the database connection object
	* @see DbHelper
	*/

	// fix! need no confs - deprecate and move function to findDemNameContains(String str, Connection connection)

	public static Vector findDemNameContains(String str, C5UTMconfs confs, Connection connection) {
		Vector<DemMetadata> vec = new Vector<DemMetadata>();
		try {

			Statement select = connection.createStatement();
			ResultSet result = select.executeQuery
				("SELECT * FROM DEM_METADATA where file_info LIKE '%" + str + "%'");

			while (result.next()) {
				vec.add(new DemMetadata(
						result.getString("DEM_METADATA_id"),
						result.getInt("zone"),
						result.getInt("columns_x"),
						result.getInt("max_profile_length_y"),
						result.getDouble("sw_easting"),
						result.getDouble("sw_northing"),
						result.getDouble("nw_easting"),
						result.getDouble("nw_northing"),
						result.getDouble("ne_easting"),
						result.getDouble("ne_northing"),
						result.getDouble("se_easting"),
						result.getDouble("se_northing"),
						result.getDate("date_added"),
						result.getString("local_dem_file"),
						result.getString("file_info"),
						result.getInt("1K_grid"),
						result.getInt("point_stats"),
						result.getDouble("average_elevation")
					)
				);
			}
			select.close();

		} catch (SQLException e) {
			System.err.println("Exception when checking for pre-existing entry.");
			e.printStackTrace();
		}
		return vec;
	}

	/** <code>Vector findDemNameContains(String str)</code>
	* This method will normally return a Vector with either 0 or 1 element
	* of type DemMetadata. It searches the file_info field of DEM_METADATA for a
	* subsring containing the query string. If you have a confs
	* object available, use <code>Vector findDemNameContains(String str, C5UTMconfs conf)</code>
	* This overloaded version is slower because it has the overhead of making a new C5UTMconfs.
	* @see com.c5corp.c5utm.DemMetadata
	* @param str the query string
	* @param connection the database connection object
	* @see DbHelper
	*/
	public static Vector findDemNameContains(String str, Connection connection) {
		C5UTMconfs confs = new C5UTMconfs();
		return findDemNameContains(str, confs, connection);
	}

	/** <code>String deleteDem(String id, C5UTMconfs conf)</code>
	* This method drops a dem from the database according to its id. The id can be found
	* in the RECEIPT folder. Also deletes the receipt file.
	* @see com.c5corp.c5utm.C5UTMconfs
	* @param id the id to delete for
	* @param confs the configuration object
	* @see DbHelper
	*/
	public static String deleteDem(String id, C5UTMconfs confs) {
		int meta_data_result = -1;
		int coords_result = -1;

		// delete DEM_METADATA record
		try {

			Connection connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());
			Statement s = connection.createStatement();
			meta_data_result = s.executeUpdate
				("DELETE FROM DEM_METADATA where DEM_METADATA_id=\"" + id + "\"");

			s.close();
			connection.close();     // close db connection

		} catch (SQLException e) {
			System.err.println("Exception when trying to delete dem id " + id + " from DEM_METADATA");
			e.printStackTrace();
			return ("Exception when trying to delete dem id " + id + " from DEM_METADATA");
		}

		// delete UTM_Coords records
		try {

			Connection connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());
			Statement s = connection.createStatement();
			coords_result = s.executeUpdate
				("DELETE FROM UTM_COORD where DEM_METADATA_id=\"" + id + "\"");

			s.close();
			connection.close();     // close db connection

		} catch (SQLException e) {
			System.err.println("Exception when trying to drop dem id " + id + " from UTM_COORD");
			e.printStackTrace();
			return ("Exception when trying to drop dem id " + id + " from UTM_COORD.");
		}


		return "Deleted DEM_METADATA_id " + id + ", " + meta_data_result + " DEM_METADATA record and "
			+ coords_result + " UTM_COORD records.";
	}

	/** <code>boolean deleteDem(String id)</code>
	* This method drops a dem from the database according to its id. The id can be found
	* in the RECEIPT folder. Also deletes the receipt file. If you have a confs
	* object available, use <code>boolean dropDem(int id, C5UTMconfs conf)</code>
	* This overloaded version is slower because it has the overhead of making a new C5UTMconfs.
	* @param id the id of the DEM to delete
	* @see DbHelper
	*/
	public static String deleteDem(String id) {
		C5UTMconfs confs = new C5UTMconfs();
		return deleteDem(id, confs);
	}
}


