package com.c5corp.c5utm;

/** <code>public class Points</code>
* <p>ArtificialPoints contains a single static method, public static Points getPoints(UtmCoordinatePairElev[][]),
* which will build a Points object from programatic data (An array of UtmCoordinatePairElev[][])
* instead of pulling an existing Points object from the relational database. This is useful for making artificial
* landscapes or creating database-like (com.c5corp.c5utm) objects from some non-database data source, such as a Dem.</p>
* @see com.c5corp.c5dem.UtmCoordinatePairElev
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0.3
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import com.c5corp.c5dem.*;

public class ArtificialPoints {

	/**
	* Builds a Points object from programatic data (An array[][] of UtmCoordinatePairElev and a utm zone)
	* instead of pulling an existing Points object from the database. This is useful for making artificial
	* landscapes or creating database-like objects from some non-database data source, such as a com.c5corp.c5dem.DemTable.</p>
	* @param coords the array of point/elevation pairs
	* @param zone the UTM zone for the area
	* @return com.c5corp.c5utm Points a Points object representing the coords and zone paramters
	* @see com.c5corp.c5dem.DemTable
	* @see com.c5corp.c5utm
	*/
	public static Points getPoints(UtmCoordinatePairElev[][] coords, int zone) {
		// first we must analyse aspects of the coords that the constructor for com.c5corp.c5utm.Points needs.
		int lowest = 100000; // higher than the highest point on earth
		int highest = -100000; // lower than the lowest point on earth
		int largest_col=-1;
		int[] shape = new int[coords.length];
		Point[][] point_arr;

		for (int i = 0; i < coords.length; i++) {
			shape[i] = coords[i].length;
			for (int j = 0; j < coords[i].length; j++) {
				if (coords[i][j] != null) {
					int elev = coords[i][j].getElevation();
					if (elev > highest) highest = elev;
					if (elev < lowest) lowest = elev;
				} else {
					System.err.println("ArtificialPoints, public static Points ArtificalPoints(UtmCoordinatePairElev[][]) " +
						"must be called with a UtmCoordinatePairElev[][] that contains no null values");
					return null;
				}
			}

			if (coords[i].length > largest_col) largest_col = coords[i].length;
		}

		// The UtmCoordinatePairElev[][] array must be converted to a com.c5corp.c5utm.Point[][] array
		point_arr = new Point[shape.length][largest_col]; 	// largest_col may cause some null values because
									// not every col is the same size

		for (int i = 0; i < coords.length; i++) {
			for (int j = 0; j < coords[i].length; j++) {
				if (coords[i][j] != null) {
					point_arr[i][j] = new Point(zone,
						coords[i][j].getEasting(),
						coords[i][j].getNorthing(),
						coords[i][j].getElevation(),
						"", // meanlinless in this context - dem_id is in the db, but this is from the dem file
						""); // meanlinless in this context - id same idea.
				}
			}
		}
		return new Points(lowest, highest, largest_col, point_arr, shape, false, 0.0); // artificial assumes no voids
	}
}
