package com.c5corp.c5dem;

import java.io.*;

/**
* <p>Object model of a Type C DEM record</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class TypeC {

	////////////////
	//DECLARATIONS//
	////////////////


	// *****Logical record C variables*****

	// Data Element 1
	private short statsavailable; /*	Code indicating availability of statistics in data element 2
					1-6
					1=available
					0=unavailable */
	// Data Element 2
	private short filermse;		/*	RMSE of file's datum relative to absolute datum (x, y, z)
					7-24
					In same units as indicated by elements 8 and 9 of logical
					record type A.  */
	// Data Element 3
	private short filermsesamplesize;	/*	Sample size on which statistics in data element 2 are based
					25-30
					If 0, then accuracy will be assumed to be estimated rather
					than computed. */
	// Data Element 4
	private short availability;	/*	Code indicating availability of statistics in data element 5
					31-36
					1=available
					0=unavailable */
	// Data Element 5
	private short datarmse; 		/* 	RMSE of DEM data relative to file's datum (x, y, z)
					37-54
					In same units as indicated by elements 8 and 9 of logical
					record type A. */
	// Data Element 6
	private short datarmsesamplesize; /*	Sample size on which statistics in element 5 are based
					55-60
					If 0, then accuracy will be assumed to be estimated rather
					than computed. */
	// This object can not be initialized until the entire type B record has been read.
	private boolean initialzed=false;

	Reader reader;

	////////////////
	//CONSTRUCTOR//
	////////////////

	/** Constructor for a TypeC object, builds itself a typeB_profiles object.*/
	TypeC (TypeBprofiles typeB_profiles) {
		// find the position of the type C record (where B left off...)
		reader=typeB_profiles.getReader();
		// Data Element 1
		statsavailable=reader.ReadDataElementsShort(1)[0];
		// Data Element 2
		filermse=reader.ReadDataElementsShort(1)[0];
		// Data Element 3
		filermsesamplesize=reader.ReadDataElementsShort(1)[0];
		// Data Element 4
		availability=reader.ReadDataElementsShort(1)[0];
		// Data Element 5
		datarmse=reader.ReadDataElementsShort(1)[0];
		// Data Element 6
		datarmsesamplesize=reader.ReadDataElementsShort(1)[0];
	}

	///////////
	//METHODS//
	///////////

	// access methods

	// Data Element 1
	/** Returns the code indicating availability of statistics in data element 2:
	1=available, 0=unavailable */
	public short get_stats_available() {
		return statsavailable;
	}

	// Data Element 2
	/**	RMSE of file's datum relative to absolute datum (x, y, z)
	In same units as indicated by elements 8 and 9 of logical
	record type A.
	@see com.c5corp.c5dem.TypeA#get_planimetric_unit
	@see com.c5corp.c5dem.Dem#get_planimetric_unit
	@see com.c5corp.c5dem.TypeA#get_elevation_unit
	@see com.c5corp.c5dem.Dem#get_elevation_unit
	*/
	public short get_file_rmse() {
		return filermse;
	}

	// Data Element 3
	/**	Sample size on which statistics in data element 2 are based
	If 0, then accuracy will be assumed to be estimated rather
	than computed. */
	public short get_file_rmse_sample_size() {
		return filermsesamplesize;
	}

	// Data Element 4
	/**	Code indicating availability of statistics in data element 5:
	1=available, 0=unavailable */
	public short get_availability() {
		return availability;
	}

	// Data Element 5
	/** RMSE of DEM data relative to file's datum (x, y, z)
	In same units as indicated by elements 8 and 9 of logical
	record type A.
	@see com.c5corp.c5dem.TypeA#get_planimetric_unit
	@see com.c5corp.c5dem.Dem#get_planimetric_unit
	@see com.c5corp.c5dem.TypeA#get_elevation_unit
	@see com.c5corp.c5dem.Dem#get_elevation_unit*/
	public short get_data_rmse() {
		return datarmse;
	}

	// Data Element 6
	/**	Sample size on which statistics in element 5 are based
	If 0, then accuracy will be assumed to be estimated rather
	than computed. */
	public short get_data_rmse_sample_size() {
		return datarmsesamplesize;
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
