package com.c5corp.c5dem;

import java.io.*;

/**
* </p>Container and reader for multiple TypeB profile objects
* This is only called by the container class Dem.
* TypeA public short getColumns() returns the number of columns</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class TypeBprofiles {

	////////////////
	//DECLARATIONS//
	////////////////

	private Reader reader;
	private TypeA typeA;
	private TypeB[] typeBarray;

	// *****Logical record B variables*****
	// vars for reading in and building profiles

	// Data Element 1
	private short[] rowandcolumnid;

	// Data Element 2
	private short[] numberofelevations;

	// Data Element 3
	private double[] firstelevationcoords;

	// Data Element 4
	private double localelevation;

	private double[] profileminmax;

	// Data Element 6
	private int[] elevations;

	// Not a data element in the DEM format, but important for imaging applications
	private short max_elevation_count_for_all_profiles;

	////////////////
	//CONSTRUCTORS//
	////////////////

	// constructor for input from an InputStream specified by a instream
	// The constructor for a TypeBprofiles object loads the TypeB profile
	// objects into the array profile array
	TypeBprofiles(TypeA typeA) {
		this.typeA=typeA;
		reader=typeA.getReader();
		int number_of_columns=typeA.get_rows_and_columns()[1];
		max_elevation_count_for_all_profiles=0;
		typeBarray=new TypeB[number_of_columns];
		for (int i=0; i<number_of_columns; i++) {
			typeBarray[i]=buildTypeB();
			// discover the max number of elevations for all TypeB profiles
			short temp=typeBarray[i].get_profile_elevations();
			if (temp > max_elevation_count_for_all_profiles) {
				max_elevation_count_for_all_profiles = temp;
			}
		}
	}

	///////////
	//METHODS//
	///////////

	// access methods

	/** Returns a TypeB object at the specified index. */
	protected TypeB getTypeB(int num) {
		return typeBarray[num];
	}

	/** Returns the maximum elevation count for all profiles. */
	protected short maxElevationsForAllProfiles() {
		return max_elevation_count_for_all_profiles;
	}

	// private utility methods

	// buildTypeB() builds a typeB using other utility methods
	private TypeB buildTypeB() {
		TypeB typeB;
		loadTypeBInfo();
		this.elevations=loadTypeBValues();
		typeB= new TypeB(this.rowandcolumnid, this.numberofelevations,
			this.firstelevationcoords, this.localelevation,
			this.profileminmax, this.elevations);
		return typeB;
	}

	// reads the profile's info (or header) variables- used by public getNextProfile()
	private void loadTypeBInfo() {

		// Element 1
		if (reader.eof == false) {
			this.rowandcolumnid=reader.ReadDataElementsShort(2);
		}

		// Element 2
		if (reader.eof == false) {
			this.numberofelevations=reader.ReadDataElementsShort(2);
		}

		// Element 3
		if (reader.eof == false) {
			this.firstelevationcoords=reader.ReadDataElementsDouble(2);
		}

		// Element 4
		if (reader.eof == false) {
			double[] temp=reader.ReadDataElementsDouble(1);
			this.localelevation=temp[0];
		}

		// Element 5
		if (reader.eof == false) {
			this.profileminmax=reader.ReadDataElementsDouble(2);
		}
	}


	// private short[] getProfileValues()
	// gets the profile values used by public getNextProfile()
	// converted from short to int in version 1.0.2
	private int[] loadTypeBValues() {

		int[] elevat;

		//Element 6
		if (reader.eof == false) {
			elevat=reader.ReadDataElementsInt(numberofelevations[0]);  // passes num of columns in this profile
		} else {
			return null;
		}
		return elevat;
	}

	// misc
	/** Method for the next TypeBprofiles objects and TypeC object to get the Reader object.*/
	public Reader getReader() {
		Reader tmp;
		tmp=reader;
		reader=null;	// make our reader null - we are done with it
		return tmp;
	}
	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
