package com.c5corp.c5dem;

import java.io.*;

/**
* <p>Object model of a Type B DEM record</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class TypeB {

	////////////////
	//DECLARATIONS//
	////////////////

	// *****Logical record B variables*****
	// these variables are recycled as necessary to read in all type b data...
	// there is no model of the entire dem in memeory... too large

	// Data Element 1
	private short[] rowandcolumnid;
			/*		A two-element array containing the row and column
					identification number of the DEM profile contained in this
					record
					1-12
					See figure 5.  The row/column numbers may range from 1 to m
					and 1 to n. The row number is normally set to 1.  The
					column identification is the profile sequence number. */
	// Data Element 2
	private short[] numberofelevations;
				/* 	A two-element array containing the number (m, n) of
					elevations in the DEM profile
					13-24
					See figure 5.  The first element in the field
					corresponds to the number of rows and columns
					of nodes in this profile.  The second
					element is set to 1, specifying 1 column per B record. */
	// Data Element 3
	private double[] firstelevationcoords;
				/* 	A two-element array containing the ground planimetric
					coordinates (Xgo,Ygo) of the first elevation in the profile
					25-72	See figure 5. */
	// Data Element 4
	private double localelevation;
				/*	Elevation of local datum for the profile
					73-96
					The values are in the units of measure given by data
					element 9, logical record type A. */
	// Data Element 5
	private double[] profileminmax;
				/* 	A two-element array of minimum and maximum elevations
					for the profile
					97-144
					The values are in the unit of measure given by data
					element 9 in logical record type A. */
	// Data Element 6
	private int[] elevations; // converted from short to int in version 1.0.2
			/*		A m,n array of elevations for the profile. Elevations
					are expressed in units of resolution.
					6x(146 or 170) 146 = max for first block.
					170 = max for subsequent blocks
					See data element 15 in appendix A. A value in this array
					would be multiplied by the spatial resolution value and
					added to the elevation of the local elevation datum for
					the element profile (data element 4 in this record) to
					obtain the elevation for the point.  The planimetric ground
					coordinates of the point Xgp, Ygp, are computed according to
					the formulas in figure 5. */

	////////////////
	//CONSTRUCTORS//
	////////////////

	// constructor gets the number of rows in the column specifed by the profile
	TypeB(short[] rowandcolumnid, short[] numberofelevations, double[] firstelevationcoords,
	double localelevation, double[] profileminmax, int[] elevations) {

		// instantiate stuff...
		this.rowandcolumnid=rowandcolumnid;
		this.numberofelevations=numberofelevations;
		this.firstelevationcoords=firstelevationcoords;
		this.localelevation=localelevation;
		this.profileminmax=profileminmax;
		this.elevations=elevations;
	}

	///////////
	//METHODS//
	///////////

	// variable access methods

	/** A two-element array representing data element 1, containing the row and column identification number of
	the DEM profile contained in this record. The row/column numbers may range from 1 to m
	and 1 to n. The row number is normally set to 1.  The
	column identification is the profile sequence number. */
	public short[] get_row_and_column_id() {
		return rowandcolumnid;
	}

	/** A convenience method for Data element 1 retruning the row id */
	public short get_row_id() {
		return rowandcolumnid[0];
	}

	/** A convenience method for Data element 1 retruning the column id */
	public short get_column_id() {
		return rowandcolumnid[1];
	}

	/**	A two-element array containing the number (m, n) of
		elevations in the DEM profile, (Data element 2). The first element in the field
		corresponds to the number of rows and columns
		of nodes in this profile.  The second element is set to 1, specifying 1 column per B record. */
	public short[] get_number_of_elevations() {
		return numberofelevations;
	}

	/**Convenience methods for Data Element 2,
	returns the number of elevations in this profile*/
	public short get_profile_elevations() {
		return numberofelevations[0];
	}

	/**Convenience methods for Data Element 2,
	returns the number of columns per profile in this file.
	This should always be 1 - at least for typical 1:24000 files.*/
	public short get_profile_columns() {
		return numberofelevations[1];
	}

	/** 	A two-element array containing the ground planimetric
	coordinates (Xgo,Ygo) of the first elevation in the profile.
	(Data element 3).
	*/
	public double[] get_first_elevation_coords() {
		return firstelevationcoords;
	}

	/** A convenience method for data element 3, the ground planimetric
	coordinates Xgo.*/
	public double get_xgo() {
		return firstelevationcoords[0];
	}

	/** A convenience method for data element 3, the ground planimetric
	coordinates Ygo.*/
	public double get_ygo() {
		return firstelevationcoords[1];
	}

	/**	Elevation of local datum for the profile
		The values are in the units of measure given by data
		element 9, logical record type A. (Data element 4)
		@see com.c5corp.c5dem.TypeA#get_elevation_unit
		@see com.c5corp.c5dem.Dem#get_elevation_unit
	*/
	public double get_datum_elevation() {
		return localelevation;
	}

	/** A two-element array of minimum and maximum elevations for the profile. (Data element 5).
		The values are in the unit of measure given by data element 9 in logical record type A.
		@see com.c5corp.c5dem.TypeA#get_elevation_unit
		@see com.c5corp.c5dem.Dem#get_elevation_unit
	*/
	public double[] get_profile_min_max() {
		return profileminmax;
	}

	/** A convenience method for data element 5, the minumum elevation in the profile.*/
	public double get_min_elevation() {
		return profileminmax[0];
	}

	/** A convenience method for data element 5, the maximum elevation in the profile.*/
	public double get_max_elevation() {
		return profileminmax[1];
	}

	/** A m,n array of elevations for the profile. (Data element 6). Elevations
	are expressed in units of resolution. 6x(146 or 170) 146 = max for first block.
	170 = max for subsequent blocks.
	See data element 15 in appendix A. A value in this array
	would be multiplied by the spatial resolution value and
	added to the elevation of the local elevation datum for
	the element profile (data element 4 in this record) to
	obtain the elevation for the point.  The planimetric ground
	coordinates of the point Xgp, Ygp, are computed according to
	the formulas in figure 5. (see dem specification, 1993)*/
	public int[] get_elevations() {
		return elevations;
	}

	/** Returns the elevation at index i
	converted to int in version 1.0.2 */
	public int get_elevation_at(short i) {
		return elevations[i];
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
