package com.c5corp.c5dem;

import java.io.*;
import java.net.*;

/**
* <p>Object model of a Type A DEM record.
* There is a great deal of conflicting infomation about the DEM format.
* The notes below are taken from the Digital Elevation Model Data Users
* Guide 5, published in 1993. It assumes that the "new" format DEM file that
* supports data elements 17 through 29. It appears to support post OCT 2001
* files, 1:24000 large scale dems. It is only intended to support USGS 7.5
* minute large scale 1:24000 dems, (utm coords only).</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class TypeA {

	////////////////
	//DECLARATIONS//
	////////////////

	// *****Logical record A variables*****

	// Data Element 1
	private String filename; /*	Authorized DEM quadrangle name
				1-40 */
				/* 	Free format descriptor field
				41-80
				contains useful information related to digital process such
				as digitizing instrument, photo codes, slot widths, etc.*/
				//Filler 81-135
				/* 	Process Code 136
				1=GPM
				2=Manual Profile
				3=DLG2DEM (includes any DLG type process such as CTOG or
				LINETRACE)
				4=DCASS */
				// Filler 137
				/* Sectional Indicator,138-140 This code is specific to
				30-minute DEM's.  Identifies 1:100,000 scale sections.
				Formatted as XNN, where X is "s"=7.5-minute,
				"F"=15-minute, and NN is a two-digit sequence number.*/
				// Overall 1-140

	// Data Element 2
	private String mcorigin; /* 	MC origin code
				141-144
				Mapping Center origin Code.
				Valid codes are	 EMC, WMC, MCMC, RMMC, FS, GPM2. */
	// Data Element 3
	private short demlevelcode; /*	DEM level code 145-50
				1=DEM-1
				2=DEM-2
				3=DEM-3 */
	// Data Element 4
	private short elevationpattern;/*	Code defining elevation pattern (regular or random) 151-156
				1=regular
				2=random is reserved for future use. */
	// Data Element 5
	private short planimetricsystem;  /*Code defining ground planimetric reference system 157-162
					0=Geographic
					1=UTM
					2=State plane
				(For codes 3-20, see appendix F, USGS DEM Data Users Guide
				Code 0 represents the geographic (latitude/longitude) system for
				30-minute, 1-degree and Alaska DEM's.
				Code 1 represents the current use of the UTM coordinate system
				for 7.5-minute DEM's. */
	// Data Element 6
	private short planimetriczone; /* Code defining zone in ground planimetric reference system 163-168
				Codes for State plane and UTM coor- dinate zones are given in
				appendixes D and E for 7.5-minute DEM's.  Code is set to zero
				if element 5 is also set to zero defining data as geographic. */
	// Data Element 7
	private double[] projectionparameters= new double[15];
				/* Map projectionparameters (see appendix F) of measure for
				ground planimetric coordinates through-out the file
				169-528
				Definition of parameters for various projections is given
				in appendix F.  All 15 fields of this element are set to
				zero and should be ignored when geographic, UTM, or
				State plane coordinates are coded in data element 5. */
	// Data Element 8
	private short planimetricunit; /*	Code defining unit of measure for ground planimetric units throughout
				the file 529-534
				0=radians
				1=feet
				2=meters
				3=arc-seconds
				Normally set to code 2 for 7.5-minute DEM's.
				Always set to code 3 for 30-minute, 1-degree, and Alaska DEM's. */
	// Data Element 9
	private short elevationunit; /* 	Code defining unit of measure for elevation coordinates
				throughout the file
				535-540
				1=feet
				2=meters
				Normally code 2, meters, for 7.5-minute, 30-minute,
				1-degree, and Alaska DEM's. */
	// Data Element 10
	private short sides; /* Number (n) of sides in the polygon which defines the coverage
				of the DEM file
				541-546
				(n=4) */
	// Data Element 11
	private double[][] corners= new double[4][2]; /* 	A 4,2 array containing the ground coordinates of the four
				corners for the DEM.
				547-738
				The coordinates of the quadrangle corners are ordered in a clockwise
 				direction beginning with the southwest corner. The array is stored
 				row-wise as pairs of eastings and northings. */
 	// Data Element 12
	private double[] minmax= new double[2]; /*	A two-element array containing minimum and maximum elevations
				for the DEM
				739-786
				The values are in the unit of measure given by data element 9 in
				this record. */
	// Data Element 13
	private double counterangle; /* Counterclockwise angle (in radians) from the primary axis of ground
				planimetric reference to the primary axis of the DEM local
				reference system
				787-810
				See figure 5.  Set to zero to align cordinate system specified
				in element 5. */
	// Data Element 14
	private short elevationaccuracy; /* Accuracy code for elevations
				811-81
				0=unknown accuracy
				1=accuracy information is given in logical record type C.*/
	// Data Element 15
	private double[] xyzresolution= new double[3]; /* A three-element array of DEM spatial resolution for x, y, z.
				Units of measure are consistent with those indicated by data
				elements 8 and 9 in this record.
				817-852
				These elements are usually set to;
				30, 30, 1 for 7.5-minute DEM's;
				2, 2, 1 for 30-minute DEM's;
				3, 3, 1 for 1-degree DEM's.
				2, 1, 1 for high resolution DEM's in Alaska
				3, 2, 1 for low resolution DEM's in Alaska.
 				7.5-minute DEM'S will eventually be converted
				to geographics, i.e., 1,1,1. */
	// Data Element 16
	private short[] rowsandcolumns=new short[2]; /* A two-element array containing the number of rows and
				columns (m,n) of profiles in the DEM
				853-864
				When the row value m is set to 1 the n value describes the number
				of columns in the DEM file.  Raw GPM data files are set to
				m=16, n=16. */


	/* Note: Old format stops here- older DEM applications and files do not
	support the Elements 17 to 24 */


	// Data Element 17
	private short largestinterval; /*		Largest primary contour interval
					865-869
					Present only if two or more
					primary intervals exist. */
	// Data Element 18
	private short contourintervalL; /*	Source contour interval units
					870
					Corresponds to the units of the map largest primary contour
					interval. 0=N.A., 1=feet, 2=meters. */
	// Data Element 19
	private short smallestprimary; /* 	Smallest primary
					871-875
					Smallest or only primary contour interval */
	// Data Element 20
	private short contourintervalS; /*	Source contour interval units
					876
					Corresponds to the units of the map smallest primary
					contour interval. 1=feet, 2=meters. */
	// Data Element 21
	private short sourcedate;
					/* Data source date
					877-880
					YYMM two-digit year and two-digit month
					MM = 00 for source having year only. */
	// Data Element 22
	private short inspectiondate; /* 		Data inspection revision date
					881-884
					YYMM two-digit year and two-digit month. */
	// Data Element 23
	private char inspectionflag; /* 	Inspection revision flag
					885
					"I" or "R" */
	// Data Element 24
	private short validationflag; /*		Data validation flag
					886
					0= No validation performed.
					1=TESDEM (record C added) no qualitative test
					(no DEM Edit System [DES] review).
					2=Water body edit and TESDEM run.
					3=DES (includes water edit) no  qualitative test (no TESDEM).
					4=DES with record C added, qualita tive and quantitative
					tests for level 1 DEM.
 					5=DES and TESDEM qualitative and quantitative tests for
 					levels 2 and 3 DEM's. */
 	// Data Element 25
	private short suspectandvoidflag; /* 	Suspect and void area flag
					887-888
					0=none
					1=suspect areas
					2=void areas
					3=suspect and void areas */
	// Data Element 26
	private short verticaldatum; /*		Vertical datum
					889-890
					1=local mean sea level
					2=National Geodetic Vertical Datum 1929 (NGVD 29)
					3=North American Vertical Datum 1988 (NAVD 88) */
	// Data Element 27
	private short horizontaldatum; /* 	Horizontal datum
					891-892
					1=North American Datum 1927 (NAD 27)
					2=World Geodetic System 1972 (WGS 72)
					3=WGS 84
					4=NAD 83
					5=Old Hawaii Datum
					6=Puerto Rico Datum
					7=NAD 83 Provisional (shifts in horizontal coordinates are
					computed, but old DEM nodes are not resampled) */
	// Data Element 28
	private short dataedition; /*		Data Edition
					893-896
					01-99 Primarily a NIMA specific field. */
	// Data Element 29
	private short percentvoid; 	/*	Percent Void
					897-900
					If element 25 indicates a void, this field (right justified)
					contains the percentage of nodes in the file set to
					void (-32,767). */

	// misc
	private Reader reader;


	////////////////
	//CONSTRUCTORS//
	////////////////

	/** Constructor for input from an InputStream specified by the path given in String instream.
	This basic constructor for a TypeA object builds itself from an filename.
	This is public because there are times when it is useful to open the
	type A data outside of this package.*/
	public TypeA(String instring) {
		try {
			reader=new Reader(instring);
		} catch(IOException ioe) {
			ioe.printStackTrace();
		}

		buildTypeA();
	}

	/** Constructor for input from an InputStream specified by the path given in String instream.
	This basic constructor for a TypeA object builds itself from an filename.
	This is public because there are times when it is useful to open the
	type A data outside of this package.*/
	public TypeA(URL url) {
		try {
			reader=new Reader(url);
		} catch(IOException ioe) {
			ioe.printStackTrace();
		}
		buildTypeA();
	}

	/* This method is utilized by the contructors in this class to read the
	* file and build the TypeA. */
	private void buildTypeA() {
		// (the DEM file format documentation assumes 1 based indexing)

		// parse the type A record
		// read records into variables
		// Element 1
		if (reader.eof == false) {
			// code to delete whitespace added to verson 1.0.2
			int end = 140;
			//filename = temp.trim();
			String temp=reader.readStringRange(1,end);
			while (temp.charAt(end-1) == ' ' && (end > 0)) {
				--end;
			}
			filename = temp.substring(0,end);

		}

		// Element 2
		if (reader.eof == false) {
			mcorigin=reader.readStringRange(141,144);
		}

		// Element 3
		if (reader.eof == false) {
			demlevelcode=reader.readShortRange(145,150);
		}

		// Element 4
		if (reader.eof == false) {
			elevationpattern=reader.readShortRange(151,156);
		}

		// Element 5
		if (reader.eof == false) {
			planimetricsystem=reader.readShortRange(157,162);
		}

		// Element 6
		if (reader.eof == false) {
			planimetriczone=reader.readShortRange(163,168);
		}

		// Element 7 (array of 15 doubles) (169 to 528)
		int k =169;
		int[][] arr=new int[15][2];
		for (int i=0; i<15; i++) {
			arr[i][0]=k;
			k+=23;
			arr[i][1]=k;
			k++;
		}
		for (int i=0; i<15; i++) {
			if (reader.eof == false) {
				projectionparameters[i]=reader.readDoubleRange(arr[i][0], arr[i][1]);
			}
		}

		// Element 8

		if (reader.eof == false) {
			planimetricunit=reader.readShortRange(529,534);
		}

		// Element 9
		if (reader.eof == false) {
			elevationunit=reader.readShortRange(535,540);
		}

		// Element 10
		if (reader.eof == false) {
			sides=reader.readShortRange(541,546);
		}

		// Element 11 (547 to 738)
		k =547;
		arr= new int[4][2];
		for (int i=0; i<4; i++) {
			arr[i][0]=k;
			k+=23;
			arr[i][1]=k;
			k++;
		}
		for (int i=0; i<4; i++) {
			if (reader.eof == false) {
				corners[i][0]=reader.readDoubleRange(arr[i][0], arr[i][1]);
			}
			if (reader.eof == false) {
				corners[i][1]=reader.readDoubleRange(arr[i][0], arr[i][1]);
			}
		}

		// Element 12 (739 to 786)
		k =739;
		arr= new int[2][2];
		for (int i=0; i<2; i++) {
			arr[i][0]=k;
			k+=23;
			arr[i][1]=k;
			k++;
		}
		if (reader.eof == false) {
			minmax[0]=reader.readDoubleRange(arr[0][0], arr[0][1]);
		}
		if (reader.eof == false) {
			minmax[1]=reader.readDoubleRange(arr[1][0], arr[1][1]);
		}

		// Element 13
		if (reader.eof == false) {
			counterangle=reader.readDoubleRange(787,810);
		}

		// Element 14
		if (reader.eof == false) {
			elevationaccuracy=reader.readShortRange(811,816);
		}

		// Element 15
		k=817;
		arr= new int[3][2];
		for (int i=0; i<3; i++) {
			arr[i][0]=k;
			k+=11;
			arr[i][1]=k;
			k++;
		}
		for (int i=0; i<3; i++) {
			if (reader.eof == false) {
				xyzresolution[i]=reader.readDoubleRange(arr[i][0],arr[i][1]);
			}
		}

		// Element 16
		if (reader.eof == false) {
			rowsandcolumns[0]=reader.readShortRange(853,858);
		}
		if (reader.eof == false) {
			rowsandcolumns[1]=reader.readShortRange(859,864);
		}

		/* Note: Old format stops here- older DEM applications and files do not
		support the Elements 17 to 24, and no attempt is made here to support
		older DEM files. I am not sure older files contain whitespace between 865
		and 900*/

		// Data Element 17
		largestinterval=reader.readShortRange(865,869);

		// Data Element 18
		contourintervalL=reader.readShortRange(870,870);

		// Data Element 19
		smallestprimary=reader.readShortRange(871,875);

		// Data Element 20
		contourintervalS=reader.readShortRange(876,876);

		// Data Element 21
		sourcedate=reader.readShortRange(877,880);

		// Data Element 22
		inspectiondate=reader.readShortRange(881,884);

		// Data Element 23
		inspectionflag=reader.readCharRange(885,885);

		// Data Element 24
		validationflag=reader.readShortRange(886,886);

	 	// Data Element 25
		suspectandvoidflag=reader.readShortRange(887,888);

		// Data Element 26
		verticaldatum=reader.readShortRange(889,890);

		// Data Element 27
		horizontaldatum=reader.readShortRange(891,892);

		// Data Element 28
		dataedition=reader.readShortRange(893,896);

		// Data Element 29
		percentvoid=reader.readShortRange(897,900);
	}

	///////////
	//METHODS//
	///////////

	/** Returns the dem spefication Data Element 1 - the file name. */
	public String get_file_name() {
		return filename;
	}

	/** Returns the dem specification Data Element 2 - MC origin code, Mapping Center origin Code.
				Valid codes are	 EMC, WMC, MCMC, RMMC, FS, GPM2.*/
	public String get_mc_origin() {
		return mcorigin;
	}

	/** Returns the dem specification Data Element 3 - DEM level code: 1=DEM-1, 2=DEM-2, 3=DEM-3 */
	public short get_dem_level_code() {
		return demlevelcode;
	}

	/** Returns the dem specification Data Element 4 - Code defining elevation pattern (regular or random):
				1=regular, 2=random is reserved for future use. */
	public short get_elevation_pattern() {
		return elevationpattern;
	}

	/** Returns the dem specification Data Element 5 - Code defining ground planimetric reference system:
	0=Geographic, 1=UTM, 2=State plane. (For codes 3-20, see appendix F, USGS DEM Data Users Guide
	Code 0 represents the geographic (latitude/longitude) system for 30-minute, 1-degree and Alaska DEM's.
	Code 1 represents the current use of the UTM coordinate system for 7.5-minute DEM's. */
	public short get_planimetric_system() {
		return planimetricsystem;
	}

	/** Returns the dem specification Data Element 6 - Code defining zone in ground planimetric reference system 163-168
	Codes for State plane and UTM coor- dinate zones are given in
	appendixes D and E for 7.5-minute DEM's.  Code is set to zero
	if element 5 is also set to zero defining data as geographic. */
	public short get_planimetric_zone() {
		return planimetriczone;
	}

	/** Returns the dem specification Data Element 7 (15 element array) -
	Map projectionparameters (see appendix F) of measure for
	ground planimetric coordinates through-out the file
	Definition of parameters for various projections is given
	in appendix F.  All 15 fields of this element are set to
	zero and should be ignored when geographic, UTM, or
	State plane coordinates are coded in data element 5. */
	public double[] get_projection_parameters() {
		return projectionparameters;
	}

	/** Returns the dem specification Data Element 8 -  Code defining unit of measure for ground planimetric units throughout
	the file: 0=radians, 1=feet, 2=meters, 3=arc-seconds
	Normally set to code 2 for 7.5-minute DEM's.
	Always set to code 3 for 30-minute, 1-degree, and Alaska DEM's. */
	public short get_planimetric_unit() {
		return planimetricunit;
	}

	/** Returns the dem specification Data Element 9 -
	Code defining unit of measure for elevation coordinates throughout the file:
	1=feet, 2=meters. Normally code 2, meters, for 7.5-minute, 30-minute, 1-degree, and Alaska DEM's. */
	public short get_elevation_unit() {
		return elevationunit;
	}

	/** Returns the dem specification Data Element 10 -
	The number (n) of sides in the polygon which defines the coverage
	of the DEM file, normally 4. */
	public short get_sides() {
		return sides;
	}

	/** Returns the dem specification Data Element 11 (a 4x2 array) -
	A 4,2 array containing the ground coordinates of the four corners for the DEM.
	The coordinates of the quadrangle corners are ordered in a clockwise
	direction beginning with the southwest corner. The array is stored
 	row-wise as pairs of eastings and northings. */
	public double[][] get_corners() {
		return corners;
	}

 	/** Returns the dem specification Data Element 12 (a 2 element array) -
 	A two-element array containing minimum and maximum elevations for the DEM.
	The values are in the unit of measure given by data element 9 in this record. */
	public double[] get_min_and_max_values() {
		return minmax;
	}

	/** Returns the dem specification Data Element 13 -
	Counterclockwise angle (in radians) from the primary axis of ground
	planimetric reference to the primary axis of the DEM local reference system.
	Set to zero to align cordinate system specified in element 5. */
	public double get_counter_angle() {
		return counterangle;
	}

	/** Returns the dem specification Data Element 14 -
	Accuracy code for elevations: 0=unknown accuracy, 1=accuracy information is given in logical record type C.*/
	public short get_elevation_accuracy() {
		return elevationaccuracy;
	}

	/** Returns the dem specification Data Element 15 (a 3 element array) -
	* A three-element array of DEM spatial resolution for x, y, z.
	Units of measure are consistent with those indicated by data
	elements 8 and 9 in this record.
	These elements are usually set to:<br>
	30, 30, 1 for 7.5-minute DEM's<br>
	2, 2, 1 for 30-minute DEM's<br>
	3, 3, 1 for 1-degree DEM's<br>
	2, 1, 1 for high resolution DEM's in Alaska<br>
	3, 2, 1 for low resolution DEM's in Alaska<br>
	7.5-minute DEM'S will eventually be converted to geographics, i.e., 1,1,1. */
	public double[] get_xyz_resolution() {
		return xyzresolution;
	}

	/** Returns the dem specification Data Element 16 -
	A two-element array containing the number of rows and columns (m,n) of profiles in the DEM
	When the row value m is set to 1 the n value describes the number
	of columns in the DEM file.  Raw GPM data files are set to
	m=16, n=16. */
	public short[] get_rows_and_columns() {
		return rowsandcolumns;
	}

	/* Note: Old format stops here- older DEM applications and files do not
	support the Elements 17 to 24 */

	/** Returns the dem specification Data Element 17 -
	Largest primary contour interval. Present only if two or more primary intervals exist. */
	public short get_largest_interval() {
		return largestinterval;
	}

	/** Returns the dem specification Data Element 18 - ource contour interval units.
	Corresponds to the units of the map largest primary contour interval. 0=N.A., 1=feet, 2=meters. */
	public short get_contour_intervalL() {
		return contourintervalL;
	}

	/** Returns the dem specification Data Element 19 -	Smallest primary.
	Smallest or only primary contour interval */
	public short get_smallest_primary() {
		return smallestprimary;
	}

	/** Returns the dem specification Data Element 20 -	Source contour interval units
	Corresponds to the units of the map smallest primary contour interval. 1=feet, 2=meters. */
	public short get_contour_intervalS() {
		return contourintervalS;
	}

	/** Returns the dem specification Data Element 21 - Data source date
	YYMM two-digit year and two-digit month. MM = 00 for source having year only. */
	public short get_source_date() {
		return sourcedate;
	}

	/** Returns the dem specification Data Element 22 -
	Data inspection revision date, YYMM two-digit year and two-digit month. */
	public int get_inspection_date() {
		return inspectiondate;
	}

	/** Returns the dem specification Data Element 23 -
	Inspection revision flag, "I" or "R" */
	public char get_inspection_flag() {
		return inspectionflag;
	}

	/** Returns the dem specification Data Element 24 -
	Data validation flag:<br>
	0= No validation performed.<br>
	1=TESDEM (record C added) no qualitative test
	(no DEM Edit System [DES] review).<br>
	2=Water body edit and TESDEM run.<br>
	3=DES (includes water edit) no  qualitative test (no TESDEM).<br>
	4=DES with record C added, qualita tive and quantitative
	tests for level 1 DEM.<br>
	5=DES and TESDEM qualitative and quantitative tests for
	levels 2 and 3 DEM's. */
	public short get_validation_flag() {
		return validationflag;
	}

 	/** Returns the dem specification Data Element 25 -
 	Suspect and void area flag: 0=none, 1=suspect areas, 2=void areas, 3=suspect and void areas */
	public short get_suspect_and_void_flag() {
		return suspectandvoidflag;
	}

	/** Returns the dem specification Data Element 26 -
	Vertical datum:<br>
	1=local mean sea level<br>
	2=National Geodetic Vertical Datum 1929 (NGVD 29)<br>
	3=North American Vertical Datum 1988 (NAVD 88) */
	public short get_vertical_datum() {
		return verticaldatum;
	}

	/** Returns the dem specification Data Element 27 -
	Horizontal datum:<br>
	1=North American Datum 1927 (NAD 27)<br>
	2=World Geodetic System 1972 (WGS 72)<br>
	3=WGS 84<br>
	4=NAD 83<br>
	5=Old Hawaii Datum<br>
	6=Puerto Rico Datum<br>
	7=NAD 83 Provisional<br>
	(shifts in horizontal coordinates are computed, but old DEM nodes are not resampled) */
	public short get_horizontal_datum() {
		return horizontaldatum;
	}

	/** Returns the dem specification Data Element 28 -
	Data Edition: 01-99 Primarily a NIMA specific field. */
	public short get_data_edition() {
		return dataedition;
	}

	/** Returns the dem specification Data Element 29 -
	Percent Void: If element 25 indicates a void, this field (right justified)
	contains the percentage of nodes in the file set to void (-32,767). */
	public short get_percent_void() {
		return percentvoid;
	}

	// misc
	/** Method for TypeBprofiles objects to get the Reader object.*/
	public Reader getReader() {
		Reader tmp;
		tmp=reader;
		reader=null;	// make our reader null - we are done with it
		return tmp;
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
