package com.c5corp.c5dem;

import java.io.*;
import java.net.URL;

/**
* <p>Reader.java is a general purpose text file reader that has
* methods for reading different data types out of the same file.
* It was designed for use with USGS DEM files and the data types
* found in that particular file format. It extends java.io.BufferedInputStream.</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class Reader extends BufferedInputStream {

	//private BufferedInputStream in; // BufferedinStream
	private int input; // the input var
	private long readerposition; // holds the readers input position (in case
	// and object with a reader wants to know...

	public boolean eof= false; // set true when eof

	//////////////////
	// CONSTRUCTORS //
	//////////////////

	/** Constructor for a reader accepts a String specifying the path to the dem file.*/
	protected Reader (String str) throws IOException {
		super(new FileInputStream(str), 8192);
		readerposition=0;
	}

	/** Constructor for a reader accepts a String specifying the path to the dem file. Allows
	the specification of a buffer size for BufferedInputStream class.*/
	protected Reader (String str, int buf) throws IOException {
		super(new FileInputStream(str), buf);
		readerposition=0;
	}

	/** Constructor for a reader accepts a java.net.URL specifying the path to the dem file. Allows
	the specification of a buffer size for BufferedInputStream class.
	@since 1.0.3b
	*/
	protected Reader (URL url) throws IOException {
			super(url.openStream());
			readerposition=0;
	}

	/** <code>public short[] ReadDataElementsShort(int number_of_elements);</code>
	A reader method that reads in a stream of Data from a Dem file
	and converts the space separated input into elements of an array...
	has ability to stop at eof (also sets eof to true) */
	public short[] ReadDataElementsShort(int number_of_elements) {
		StringBuffer thebuf=new StringBuffer();
		short[] srt=new short[number_of_elements];
		int i=0;
		boolean foundinput=false;

		while (i<number_of_elements) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					while ((char)input !=' '){
						if (input != -1) {
							thebuf.append((char)input);
							foundinput=true;
						} else {
							eof=true;
							break;
						}
						input=read();
						readerposition++;
					}
					if (foundinput) {
						srt[i]=(short)Integer.parseInt(thebuf.toString());
						foundinput=false;
						thebuf=new StringBuffer();
						i++;
					}
				}else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}
		return srt;
	}


	/** <code>public int[] ReadDataElementsInt(int number_of_elements);</code>
	reader that reads in a stream of Data from a Dem file
	and converts the space separated input into elements of an array...
	has ability to stop at eof (also sets eof to true). Added in version 1.0.2
	because dem files denominated in meters actually contain decimeters*/
	public int[] ReadDataElementsInt(int number_of_elements) {
		StringBuffer thebuf=new StringBuffer();
		int[] arr=new int[number_of_elements];
		int i=0;
		boolean foundinput=false;

		while (i<number_of_elements) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					while ((char)input !=' '){
						if (input != -1) {
							thebuf.append((char)input);
							foundinput=true;
						} else {
							eof=true;
							break;
						}
						input=read();
						readerposition++;
					}
					if (foundinput) {
						arr[i]=Integer.parseInt(thebuf.toString());
						foundinput=false;
						thebuf=new StringBuffer();
						i++;
					}
				}else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}
		return arr;
	}

	/** <code>public double[] ReadDataElementsDouble(int number_of_elements);</code>
	reader that reads in a stream of Data from a Dem file
	and converts the space separated input into elements of an array...
	has ability to stop at eof (also sets eof to true) */
	public double[] ReadDataElementsDouble(int number_of_elements) {
		StringBuffer thebuf=new StringBuffer();
		double[] dbl=new double[number_of_elements];
		int i=0;
		boolean foundinput=false;
		Double dblobj;

		while (i<number_of_elements) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					while ((char)input !=' '){
						if (input != -1) {
							if (((char)input == 'd') || ((char)input == 'D')) {
								thebuf.append('E');
							} else {
								thebuf.append((char)input);
							}
							foundinput=true;
						} else {
							eof=true;
							break;
						}
						input=read();
						readerposition++;
					}
					if (foundinput) {
						dblobj=new Double(thebuf.toString());
						dbl[i]=dblobj.doubleValue();
						foundinput=false;
						thebuf=new StringBuffer();
						i++;
					}
				}else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}
		return dbl;
	}

	/** <code>public String readStringRange(int from, int to);</code>
	reader that takes the current count from in, grabs the chars in the
	range public int from (which is the previous to) int to
	if there is no input (just space), a single space is returned */
	public String readStringRange(int from, int to) {
		StringBuffer thebuf=new StringBuffer();

		for (from=from; from<=to; from++) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					thebuf.append((char)input);
				} else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}

		return thebuf.toString();

	}

	/** <code>public char readCharRange(InputStream in, int from, int to);</code>
	reader that counts from (from) to in, grabing the chars in the
	range public int from (which is the previous to) int to
	if there is no input (just space), a single space is returned */
	public char readCharRange(int from, int to) {
		StringBuffer thebuf=new StringBuffer();
		StringBuffer temp= new StringBuffer();
		boolean found= false;

		// get it
		for (from=from; from<=to; from++) {
			try {
				thebuf.append((char)read());
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}

		// clean it (check for valid input)
		for (from=from; from<=to; from++) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					thebuf.append((char)input);
				} else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}

		// return it
		if (!found) {
			temp.append(' ');
		}
		return temp.charAt(0);
	}

	/** <code>public short readShortRange(InputStream in, int from, int to)</code>
	reader that takes the current count from in, grabs the chars in that
	range, and if there is no input (just space), -1 is returned*/
	public short readShortRange(int from, int to) {
		StringBuffer thebuf= new StringBuffer();
		StringBuffer temp= new StringBuffer();
		boolean found= false;

		// get it
		for (from=from; from<=to; from++) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					thebuf.append((char)input);
				} else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}

		// clean it
		for (short i=0; i<thebuf.length(); i++) {
			if (thebuf.charAt(i) != ' ') {
				temp.append(thebuf.charAt(i));
				found= true;
			}
		}

		// return it
		if (!found) {
			return ((short)-1);
		}
		return (short)Integer.parseInt(temp.toString());
	}

	/** public double readDoubleRangeRange(int from, int to)
	reader that takes the current count from in, grabs the chars in that
	range, and if there is no input (just space), NaN is returned.*/
	public double readDoubleRange (int from, int to) {
		StringBuffer thebuf= new StringBuffer();
		StringBuffer temp= new StringBuffer();
		double value;
		boolean found= false;

		// get it
		for (from=from; from<=to; from++) {
			try {
				input=read();
				readerposition++;
				if (input != -1) {
					thebuf.append((char)input);
				} else {
					eof=true;
					break;
				}
			} catch(IOException ioe) {
				ioe.printStackTrace();
			}
		}

		// clean it
		for (short i=0; i<thebuf.length(); i++) {
			if (thebuf.charAt(i) != ' ') {
				found= true;
				// check num format, convert to java double ('E' not 'D')
				if (thebuf.charAt(i) == 'D' || thebuf.charAt(i) == 'd') {
					temp.append('E');
				} else {
					temp.append(thebuf.charAt(i));
				}
			}
		}

		// return it
		if (!found) {
			return (Double.NaN);
		} else {
			Double dub= new Double(temp.toString());
			value=dub.doubleValue();
		}
		return value;
	}

	public long getReaderPosition() {
		return readerposition;
	}

	// misc
	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + "\n" + C5DemConstants.copy;
	}
}
