package com.c5corp.c5dem;

import java.io.*;
import java.awt.Point;
import java.net.*;

/**
* Extention of com.c5corp.c5dem.Dem adding a table of UtmCoordinatePairElev objects
* describing xy coordinates for a Dem.
* Interpolates UTM coordinates over a java image compatible
* 2D array of UtmCoordinatePairElev objects, with null values for points
* not repesented in the DEM grid).
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class DemTable extends Dem {

	//////////////////
	// DECLARATIONS //
	//////////////////

	// dem data
	private PreciseUTMcoordPair sw_corner;
	private PreciseUTMcoordPair nw_corner;
	private PreciseUTMcoordPair ne_corner;
	private PreciseUTMcoordPair se_corner;
	private short elevation_unit; // we keep a local copy of this so it can be reset...

	// table
	private UtmCoordinatePairElev[][] table;

	//////////////////
	// CONSTRUCTORS //
	//////////////////

	/** Constructs a DEM from a java.net.URL
	@see java.net.URL
	@since 1.0.3b
	*/
	public DemTable (URL url) {
		super(url);
		buildTable();
	}

	/** Constructs a DEM from an input file specified in <code>String inputfile</code>*/
	public DemTable(File file) {
		// use DemTable(String) constructor
		super(file);
		buildTable();
	}

	/** Constructs a DEM from an input file specified in <code>java.io.File file</code>*/
	public DemTable(String inputfile) {
		super(inputfile);
		buildTable();
	}

	/* This method does the work of building the DemTable, and is called by all of the constructors.*/
	private void buildTable() {

		elevation_unit = super.get_elevation_unit();

		// useful temp values to avoid calling the accessor methods
		// thousands of times...
		double x_rez=get_spacial_rez_x();
		double y_rez=get_spacial_rez_y();
		double z_rez=get_spacial_rez_z();

		// local vars in constructor
		TypeB typeB;

		// check some things, throw exceptions if unsupported
		if (get_planimetric_system() != 1) {
			throw new IllegalArgumentException
				("Unknown planimetric system: DEMtable only understands UTM");
		}

		// check planimetric unit - right now only meters
		switch (get_planimetric_unit()){
			case 3: throw new IllegalArgumentException
				("DemTable.java unfinished code: Need to implement arc second to meter conversion, (found arc second file)");
			case 1: throw new IllegalArgumentException
				("DemTable.java unfinished code: Need to implement feet to meter planimetric conversion. " +
				"(found feet file... which are not known to exist by this programmer.");
			case 2: // do nothing ... System.out.println("Found planimetric unit: meters");
				break;
			default: throw new IllegalArgumentException
				("Unknown planimetric unit");
		}
		// check elevation unit - feet or meters
		switch (get_elevation_unit()){
			case 1: // // do nothing ... System.out.println("Found elevation unit: feet");
					// throw new IllegalArgumentException // - 7/15/2k2 - *** This is the only change to Version 1.0.1
					//("DemTable.java unfinished code: Need to implement feet to meter elevation conversion. (found feet file)");
			case 2: // do nothing ... System.out.println("Found elevation unit: meters");
				break;
			default: throw new IllegalArgumentException
				("Unknown elevation unit");
		}

		// get corners - create UTM pairs
		double[][] coords=get_ground_coordinates();
		sw_corner = new PreciseUTMcoordPair(coords[0][0],coords[0][1]);
		nw_corner = new PreciseUTMcoordPair(coords[1][0],coords[1][1]);
		ne_corner = new PreciseUTMcoordPair(coords[2][0],coords[2][1]);
		se_corner = new PreciseUTMcoordPair(coords[3][0],coords[3][1]);

		// generate UTM coordinates table
		// There is no way to determine (yet) the exact 'y' size of the table
		// (because we have not read all of the TypeB records yet), but we can make
		// a guess that should be really close.
		int col_count=get_column_count();
		double northmost_northing = ne_corner.getNorthing() > nw_corner.getNorthing() ? ne_corner.getNorthing() : nw_corner.getNorthing();
		double southmost_northing = se_corner.getNorthing() < sw_corner.getNorthing() ? se_corner.getNorthing() : sw_corner.getNorthing();
		double max_northing_range = northmost_northing - southmost_northing;
		int col_length = (int)(max_northing_range / y_rez) + 1;
		table=new UtmCoordinatePairElev[col_count][col_length];

		int smallest_northing = 2000000000; // we will need this value to rearrange the data
		int[] col_sizes = new int[col_count];
		// iterate over columns
		for (int i=0; i<col_count; i++) {
			// get the next typeB and extact elevations
			typeB=getTypeB(i);
			int[] elevations = typeB.get_elevations(); // converted to int in version 1.0.2
			// save the size of the array in a parallel array
			col_sizes[i] = elevations.length;

			// Create and add UtmCoordinatePairElev objects to table
			// iterate over elevations in profile(columns) and add the data
			// the colums will not yet be correctly alligned after this process, so
			// watch for the lowest actual northing too.

			for (int j=0;j<elevations.length;j++) {
				int northing = (int)(typeB.get_ygo() + (j * y_rez)); 	// northing is interpolated according to j
											// see documentation for data element 6, TypeB
				if (northing < 	smallest_northing) smallest_northing = northing;
				table[i][j]=
					new UtmCoordinatePairElev (
						// UTM coords
						(int)typeB.get_xgo(), // easting is in profile data
							northing,
								(int)((elevations[j] + typeB.get_datum_elevation()) * z_rez) );
			}
		}

		// ok, the data is in there, but the northings are not alligned
		// allign the smallest northing throughout
		for (int i=0; i<table.length; i++) {
			// calculate the starting index offset
			int local_northing = table[i][0].getNorthing();
			int offset = (int)((local_northing - smallest_northing)/y_rez);

			UtmCoordinatePairElev[] adjusted = new UtmCoordinatePairElev[col_length];

			for (int j=offset;j<table[i].length;j++) {
				adjusted[j] = table[i][j-offset];
			}

			table[i] = adjusted;

			// reverse the elevations (java coordinate system adjustment +y is down...)
			table[i]=reverse(table[i]);

		}
	}

	/////////////
	// METHODS //
	/////////////

	/** Returns the table of UTMcoodinatePairElev points for the dem.
	*/
	public UtmCoordinatePairElev[][] getTable() {
		return table;
	}

	/** returns a UtmCoordinatePairElev[][] object where the sw_corner is easting, northing, the
	* size of easting_meters east and northing_meters north. This method returns null if it can not
	* return the full array for the area requested.
	* Added in version 1.0.3b
	*/
	public UtmCoordinatePairElev[][] getPoints(int easting, int northing, int easting_meters, int northing_meters) {
		// locals

		UtmCoordinatePairElev[][] result;
		int easting_index = -1;
		int northing_index = -1;
		int rez_x = (int)get_spacial_rez_x();
		int rez_y = (int)get_spacial_rez_y();
		int offset_x;
		int offset_y;

		// find the sw corner easting and northing closest to x,y location in table
		OUTER: for (int i = 0; i < table.length; i++) {
			for (int j = table[i].length-1; j >= 0; j--) {
				if (table[i][j] != null) {
					if ((Math.abs(table[i][j].getEasting()-easting) <= (rez_x/2))  &&
						(Math.abs(table[i][j].getNorthing()-northing) <= (rez_y/2))
					) {
						easting_index=i;
						northing_index = j;
						break OUTER;
					}
				}
			}
		}

		// instantiate result object
		offset_x = easting_meters/rez_x+1;
		offset_y = northing_meters/rez_y+1;
		result = new UtmCoordinatePairElev[offset_x][offset_y];
		// fill the result object
		// determine if all of the requested data (easting_meters, northing_meters) exists in table
		// (if not return null)
		try {
			for (int i = 0; i < offset_x; i++) {
				for (int j = 0; j < offset_y; j++) {
					//System.out.print(table[i][j] + " ");
					if (table[i+easting_index][northing_index-j] == null) return null; // never send back any nulls
					result[i][j]=table[i+easting_index][northing_index-j];
				}
				//System.out.println();
			}
		} catch (ArrayIndexOutOfBoundsException e) {
			return null; // exactly what we promise if the data is incomplete
		}

		return result;
	}

	/** public UtmCoordinatePairElev getUTMdataPoint(Point point) </code>
	* returns a UtmCoordinatePairElev object from table
	* representing the Point.
	* @deprecated This method is deprecated, its functionality replaced by the
	* <code>public UtmCoordinatePairElev getPoint(Point point) </code> method.
	*/
	// returns a UtmCoordinatePairElev object from table
	// representing the Point. Deprecated, should use getPoint(Point point)
	public UtmCoordinatePairElev getUTMdataPoint(Point point) {
		return table[point.x][point.y];
	}

	/** public UtmCoordinatePairElev getUTMdataPoint(int x, int y) </code>
	* returns a UtmCoordinatePairElev object from table
	* representing the x and y.
	* @deprecated This method is deprecated, its functionality replaced by the
	* <code>public UtmCoordinatePairElev getPoint(short x, short y) </code> method.
	*/
	public UtmCoordinatePairElev getUTMdataPoint(int x, int y) {
			return table[x][y];
	}

	/** public UtmCoordinatePairElev getUTMdataPoint(Point point) </code>
	* returns a UtmCoordinatePairElev object from table
	* representing the Point.
	*/
	public UtmCoordinatePairElev getPoint(Point point) {
		return table[point.x][point.y];
	}

	/** public UtmCoordinatePairElev getUTMdataPoint(short x, short y) </code>
	* returns a UtmCoordinatePairElev object from table
	* representing the x and y.
	*/
	public UtmCoordinatePairElev getPoint(short x, short y) {
		return table[(int)x][(int)y];
	}

	/**
	* Teturns a UtmCoordinatePairElev object from table
	* representing the closest point (snapped to grid) to the easting and northing provided.
	* The UtmCoordinatePairElev returned will always be snapped to the closest point <b>greater
	* than</b> the easting and northing points. Returns a null value if it can not find a point
	* snaps to the dem file.
	*/
	public UtmCoordinatePairElev getPoint(int easting, int northing) {
		// locals
		int rez_x = (int)get_spacial_rez_x();
		int rez_y = (int)get_spacial_rez_y();

		for (int i = 0; i < table.length; i++) {
			for (int j = table[i].length-1; j >= 0; j--) {
				//System.out.println("John C. Fremont");
				if (table[i][j] != null) {
					if (table[i][j].getEasting() >= easting &&
							table[i][j].getEasting()-easting < rez_x &&
							table[i][j].getNorthing() >= northing &&
							table[i][j].getNorthing()-northing < rez_y ) {
							// System.out.println("test: " + table[i][j] + " " + easting + " " + northing);
						return table[i][j];
					}
				}
			}
		}

		return null;
	}

	// these are just here because they might be useful later...
	// access corner objects
	public PreciseUTMcoordPair getSWcorner() {
		return sw_corner;
	}

	public PreciseUTMcoordPair getNWcorner() {
		return nw_corner;
	}

	public PreciseUTMcoordPair getNEcorner() {
		return ne_corner;
	}

	public PreciseUTMcoordPair getSEcorner() {
		return se_corner;
	}

	/**
	* public short get_elevation_unit() is a method of Dem.java that
	* is overridden in this class. <code>public void convertToVerticalFeet()</code> and
	* <code>public void convertToVerticalMeters()</code> toggle this value to represent
	* feet (1) or meters (2) after converting the values in the table to
	* the indicated unit.
	*/
	public short get_elevation_unit() {
		return elevation_unit;
	}

	/**
	* Returns a java.awt.Point describing the x,y location of the given UtmCoordinatePair
	* target within the tabular representation of the points.
	* Added in version 1.0.3b
	* @see #getTable()
	*/
	public java.awt.Point getXYofUtmPoint(UtmCoordinatePair target) {
		// get the closest UtmCoordinatePairElev in the dem table
		UtmCoordinatePairElev realTarget = getPoint(target.getEasting(), target.getNorthing());
		java.awt.Point point = null;
		FOUND: for (int i = 0; i < table.length; i++) {
			for (int j = 0; j < table.length; j++) {
				if (table[i][j] != null) {
					if (realTarget == table[i][j]) {
						point = new java.awt.Point(i,j);
						break FOUND;
					}
				}
			}
		}
		return point;
	}

	/**
	* covertToVerticalFeet() changes the internal representation of data in the dem
	* from Meters to Feet, (or leaves the units unchanged if they are already in Feet.
	* see <code>public short get_elevation_unit()</code>, a method of Dem.java that
	* is overridden in this class. <code>public short get_elevation_unit()</code> returns
	* 1 if the vertical units are in feet.
	*/
	public void convertToVerticalFeet() {
		if (elevation_unit == 1)  { // already feet, do nothing
			return;
		} else if (elevation_unit == 2) { // convert meters to feet
			for (int i = 0; i < table.length; i++) {
				for (int j = 0; j < table[i].length; j++) {
					if (table[i][j] != null) {
						table[i][j] = new UtmCoordinatePairElev(
							table[i][j].getEasting(),
							table[i][j].getNorthing(),
							(int)Math.round(table[i][j].getElevation() * C5DemConstants.METERSTOFEET)
						);
					}
				}
			}
			elevation_unit = 1;
			return;
		} else {
			throw new IllegalStateException
				("DemTable.java only understands vertical units in FEET or METERS: unknown unit found.");
		}
	}

	/**
	* covertToVerticalMeters() changes the internal representation of data in the dem
	* from Feet to Meters, (or leaves the units unchanged if they are already in Meters.
	* see <code>public short get_elevation_unit()</code>, a method of Dem.java that
	* is overridden in this class. <code>public short get_elevation_unit()</code> returns
	* 2 if the vertical units are in Meters.
	*/
	public void convertToVerticalMeters() {
		if (elevation_unit == 2)  { // already meters, do nothing
			return;
		} else if (elevation_unit == 1) { // convert feet to meters
			for (int i = 0; i < table.length; i++) {
				for (int j = 0; j < table[i].length; j++) {
					if (table[i][j] != null) {
						table[i][j] = new UtmCoordinatePairElev(
							table[i][j].getEasting(),
							table[i][j].getNorthing(),
							(int)Math.round(table[i][j].getElevation() * C5DemConstants.FEETTOMETERS)
						);
					}
				}
			}
			elevation_unit = 2;
			return;
		} else {
			throw new IllegalStateException
				("DemTable.java only understands vertical units in FEET or METERS: unknown unit found.");
		}
	}

	// private helper reverses a column to represent it properly in the data structure
	private UtmCoordinatePairElev[] reverse(UtmCoordinatePairElev[] arr) {
		UtmCoordinatePairElev[] reversed = new UtmCoordinatePairElev[arr.length];
		for (int i = 0; i < arr.length; i++) {
			reversed[i]=arr[arr.length-i-1];
		}
		return reversed;
	}


	/**
	* averageDemElevation() calculates the average elevation of this area.
	* @return the average elevation
	* @since 2.0
	*/
	public double averageDemElevation(){
		UtmCoordinatePairElev[][] tempTable = getTable();
  		if(tempTable !=null ) {
			int sum=0;
			int count=0;
			for(int a = 0; a<tempTable.length; a++) {
				for(int b = 0; b<tempTable[a].length; b++){
			    	    if(tempTable[a][b] !=null ) {
					sum += tempTable[a][b].getElevation();
					count++;

				    }
				}
			}
		return ((double)sum/(double)count);
  		}
		return -1;
	}
}
