package com.c5corp.c5dem;
import java.io.*;
import java.net.*;

/**
* <p>Digital Elevation Model (DEM) object</p>
* <p>Dem encapsulates objects for each data type in a DEM file: type a, type b, type c.
* Type a and c are single records, and are held in variables in
* a TypeA and TypeC object. TypeB records (elevation profiles) are
* contained in a TypeBprofiles object as an array. There are N profiles
* per Dem file.</p>
* <p>
* I have rewritten the interface a great deal and at different times.
* a stylistic convention I have emerged is to use shell_style method
* names for native DEM data elements, and humpNotation for generated
* metadata and other things not specified by the DEM format.
* I have not extended these to TypeA or TypeC.
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

public class Dem {

	////////////////
	//DECLARATIONS//
	////////////////
	private TypeA typeA;
	private TypeBprofiles typeB_profiles;
	private TypeC typeC;
	// private int id=-1; // not implemented
	private String inputfile;

	////////////////
	//CONSTRUCTORS//
	////////////////

	/** Constructs a DEM from an input file specified in <code>String inputfile</code>*/
	public Dem(String inputfile) {
		this.inputfile = inputfile;
		typeA= new TypeA(inputfile);
		typeB_profiles= new TypeBprofiles(typeA);
		typeC= new TypeC(typeB_profiles);
	}

	/** Constructs a DEM from an input file specified in <code>java.io.File file</code>*/
	public Dem(File file) {
		typeA= new TypeA(file.getAbsolutePath());
		typeB_profiles= new TypeBprofiles(typeA);
		typeC= new TypeC(typeB_profiles);
	}

	/** Constructs a DEM from a java.net.URL
	@see java.net.URL
	@since 1.0.3b
	*/
	public Dem(URL url) {
		typeA= new TypeA(url);
		typeB_profiles= new TypeBprofiles(typeA);
		typeC= new TypeC(typeB_profiles);
	}

	//////////////////////////////////////////
	// methods for accessing Dem Variables	//
	//////////////////////////////////////////

	// database indentifier - not implemented for anything yet
	//public int getId() {
	//	return id;
	//}

	/** Returns the name of the input file */
	// get the input file name - added 9/28/2K2
	public String getInputFile() {
		return inputfile;
	}

	// Type a methods

	/** Returns the dem spefication Data Element 1 - the file name. */
	public String get_file_name() {
		return typeA.get_file_name();
	}

	/** Returns the dem specification Data Element 2 - MC origin code, Mapping Center origin Code.
		Valid codes are	 EMC, WMC, MCMC, RMMC, FS, GPM2.*/
	public String get_mc_origin() {
		return typeA.get_mc_origin();
	}

	/** Returns the dem specification Data Element 3 - DEM level code: 1=DEM-1, 2=DEM-2, 3=DEM-3 */
	public short get_dem_level_code() {
		return typeA.get_dem_level_code();
	}

	/** Returns the dem specification Data Element 4 - Code defining elevation pattern (regular or random):
				1=regular, 2=random is reserved for future use. */
	public short get_elevation_pattern() {
		return typeA.get_elevation_pattern();
	}

	/** Returns the dem specification Data Element 5 - Code defining ground planimetric reference system:
		0=Geographic, 1=UTM, 2=State plane. (For codes 3-20, see appendix F, USGS DEM Data Users Guide
		Code 0 represents the geographic (latitude/longitude) system for 30-minute, 1-degree and Alaska DEM's.
	Code 1 represents the current use of the UTM coordinate system for 7.5-minute DEM's. */
	public short get_planimetric_system() {
		return typeA.get_planimetric_system();
	}

	/** Returns the dem specification Data Element 6 - Code defining zone in ground planimetric reference system 163-168
	Codes for State plane and UTM coor- dinate zones are given in
	appendixes D and E for 7.5-minute DEM's.  Code is set to zero
	if element 5 is also set to zero defining data as geographic. */
	public short get_planimetric_zone() {
		return typeA.get_planimetric_zone();
	}

	/** Returns the dem specification Data Element 7 (15 element array) -
	Map projectionparameters (see appendix F) of measure for
	ground planimetric coordinates through-out the file
	Definition of parameters for various projections is given
	in appendix F.  All 15 fields of this element are set to
	zero and should be ignored when geographic, UTM, or
	State plane coordinates are coded in data element 5. */
	public double[] get_projection_parameters() {
		return typeA.get_projection_parameters();
	}

	/** Returns the dem specification Data Element 8 -  Code defining unit of measure for ground planimetric units throughout
	the file: 0=radians, 1=feet, 2=meters, 3=arc-seconds
	Normally set to code 2 for 7.5-minute DEM's.
	Always set to code 3 for 30-minute, 1-degree, and Alaska DEM's. */
	public short get_planimetric_unit() {
		return typeA.get_planimetric_unit();
	}

	/** Returns the dem specification Data Element 9 -
	Code defining unit of measure for elevation coordinates throughout the file:
	1=feet, 2=meters. Normally code 2, meters, for 7.5-minute, 30-minute, 1-degree, and Alaska DEM's. */
	public short get_elevation_unit() {
		return typeA.get_elevation_unit();
	}

	/** Returns the dem specification Data Element 10 -
	The number (n) of sides in the polygon which defines the coverage
	of the DEM file, normally 4. */
	public short get_polygon_sides() {
		return typeA.get_sides();
	}

	/** Returns the dem specification Data Element 11 (a 4x2 array) -
	A 4,2 array containing the ground coordinates of the four corners for the DEM.
	The coordinates of the quadrangle corners are ordered in a clockwise
	direction beginning with the southwest corner. The array is stored
 	row-wise as pairs of eastings and northings. */
	public double[][] get_ground_coordinates() {
		return typeA.get_corners();
	}

 	/** Returns the dem specification Data Element 12 (a 2 element array) -
 	A two-element array containing minimum and maximum elevations for the DEM.
	The values are in the unit of measure given by data element 9 in this record. */
	public double[] get_min_and_max_values() {
		return typeA.get_min_and_max_values();
	}

 	/** Convienence method for Data Element 12 (which is a 2 element array), returing
 	the minimum elevation in a dem file.
	The values are in the unit of measure given by data element 9 in this record. */
	public double get_min_elevation() {
		return typeA.get_min_and_max_values()[0];
	}

 	/** Convienence method for Data Element 12 (which is a 2 element array), returing
 	the maximum elevation in a dem file.
	The values are in the unit of measure given by data element 9 in this record. */
	public double get_max_elevation() {
			return typeA.get_min_and_max_values()[1];
	}

	/** Returns the dem specification Data Element 13 -
	Counterclockwise angle (in radians) from the primary axis of ground
	planimetric reference to the primary axis of the DEM local reference system.
	Set to zero to align cordinate system specified in element 5. */
	public double get_counter_angle() {
		return typeA.get_counter_angle();
	}

	/** Returns the dem specification Data Element 14 -
	Accuracy code for elevations: 0=unknown accuracy, 1=accuracy information is given in logical record type C.*/
	public short get_accuracy_code() {
		return typeA.get_elevation_accuracy();
	}

	/** Returns the dem specification Data Element 15 (a 3 element array) -
		A three-element array of DEM spatial resolution for x, y, z.
		Units of measure are consistent with those indicated by data
		elements 8 and 9 in this record.
		These elements are usually set to:<br>
		30, 30, 1 for 7.5-minute DEM's<br>
		2, 2, 1 for 30-minute DEM's<br>
		3, 3, 1 for 1-degree DEM's<br>
		2, 1, 1 for high resolution DEM's in Alaska<br>
		3, 2, 1 for low resolution DEM's in Alaska<br>
	7.5-minute DEM'S will eventually be converted to geographics, i.e., 1,1,1. */
	public double[] get_xyz_resolution() {
		return typeA.get_xyz_resolution();
	}

	/** convenience method for element 15 - returns spacial resolution x.*/
	public double get_spacial_rez_x() {
		return typeA.get_xyz_resolution()[0];
	}

	/** convenience method for element 15 - returns spacial resolution y.*/
	public double get_spacial_rez_y() {
		return typeA.get_xyz_resolution()[1];
	}

	/** convenience method for element 15 - returns spacial resolution z.*/
	public double get_spacial_rez_z() {
		return typeA.get_xyz_resolution()[2];
	}

	/** Returns the dem specification Data Element 16 -
	A two-element array containing the number of rows and columns (m,n) of profiles in the DEM
	When the row value m is set to 1 the n value describes the number
	of columns in the DEM file.  Raw GPM data files are set to
	m=16, n=16. */
	public short[] get_rows_and_columns() {
		return typeA.get_rows_and_columns();
	}

	// Convenience method for Data Element 16
	// the number of columns in the file is the same as the number of B records
	public short get_column_count() {
			return typeA.get_rows_and_columns()[1];
	}

	/* Note: Old format stops here- older DEM applications and files do not
	support the Elements 17 to 24 */

	/** Returns the dem specification Data Element 17 -
	Largest primary contour interval. Present only if two or more primary intervals exist. */
	public short get_largest_interval() {
		return typeA.get_largest_interval();
	}

	/** Returns the dem specification Data Element 18 - ource contour interval units.
	Corresponds to the units of the map largest primary contour interval. 0=N.A., 1=feet, 2=meters. */
	public short get_contour_interval_l() {
		return typeA.get_largest_interval();
	}

	/** Returns the dem specification Data Element 19 -	Smallest primary.
	Smallest or only primary contour interval */
	public short get_smallest_primary() {
		return typeA.get_smallest_primary();
	}

	/** Returns the dem specification Data Element 20 -	Source contour interval units
	Corresponds to the units of the map smallest primary contour interval. 1=feet, 2=meters. */
	public short get_contour_interval_s() {
		return typeA.get_contour_intervalS();
	}

	/** Returns the dem specification Data Element 21 - Data source date
	YYMM two-digit year and two-digit month. MM = 00 for source having year only. */
	public short get_source_date() {
		return typeA.get_source_date();
	}

	/** Returns the dem specification Data Element 22 -
	Data inspection revision date, YYMM two-digit year and two-digit month. */
	public int get_revision_date() {
		return typeA.get_inspection_date();
	}

	/** Returns the dem specification Data Element 23 -
	Inspection revision flag, "I" or "R" */
	public char get_inspection_flag() {
		return typeA.get_inspection_flag();
	}

	/** Returns the dem specification Data Element 24 -
	Data validation flag:<br>
	0= No validation performed.<br>
	1=TESDEM (record C added) no qualitative test
	(no DEM Edit System [DES] review).<br>
	2=Water body edit and TESDEM run.<br>
	3=DES (includes water edit) no  qualitative test (no TESDEM).<br>
	4=DES with record C added, qualita tive and quantitative
	tests for level 1 DEM.<br>
	5=DES and TESDEM qualitative and quantitative tests for
	levels 2 and 3 DEM's. */
	public short get_validation_flag() {
		return typeA.get_validation_flag();
	}

 	/** Returns the dem specification Data Element 25 -
 	Suspect and void area flag: 0=none, 1=suspect areas, 2=void areas, 3=suspect and void areas */
	public short get_suspect_and_void_flag() {
		return typeA.get_suspect_and_void_flag();
	}

	/** Returns the dem specification Data Element 26 -
	Vertical datum:<br>
	1=local mean sea level<br>
	2=National Geodetic Vertical Datum 1929 (NGVD 29)<br>
	3=North American Vertical Datum 1988 (NAVD 88) */
	public short get_vertical_datum() {
		return typeA.get_vertical_datum();
	}

	/** Returns the dem specification Data Element 27 -
	Horizontal datum:<br>
	1=North American Datum 1927 (NAD 27)<br>
	2=World Geodetic System 1972 (WGS 72)<br>
	3=WGS 84<br>
	4=NAD 83<br>
	5=Old Hawaii Datum<br>
	6=Puerto Rico Datum<br>
	7=NAD 83 Provisional<br>
	(shifts in horizontal coordinates are computed, but old DEM nodes are not resampled) */
	public short get_horizontal_datum() {
		return typeA.get_horizontal_datum();
	}

	/** Returns the dem specification Data Element 28 -
	Data Edition: 01-99 Primarily a NIMA specific field. */
	public short get_data_edition() {
		return typeA.get_data_edition();
	}

	/** Returns the dem specification Data Element 29 -
	Percent Void: If element 25 indicates a void, this field (right justified)
	contains the percentage of nodes in the file set to void (-32,767). */
	public short get_percent_void() {
		return typeA.get_percent_void();
	}

	/** This method represents data not contained in the DEM format explicitly,
	but that is calculated and which is important to imaging applications.
	It is the size of the largest TypeB profile, which could be the height of
	of an image depicting the DEM, for example.
	@see com.c5corp.c5dem.TypeB
	*/
	public short maxElevationsForAllProfiles() {
		return typeB_profiles.maxElevationsForAllProfiles();
	}

	// type b methods

	/** Returns the TypeB profile indexed by num. Be sure to check typeA.get_columns(),
	(the number of profiles in the dem) or you will go out of bounds.
	@see com.c5corp.c5dem.TypeB
	*/
	public TypeB getTypeB(int num) {
		return typeB_profiles.getTypeB(num);
	}

	// type c access methods

	// Data Element 1
	/** Returns the code indicating availability of statistics in data element 2:
	1=available, 0=unavailable */
	public short get_stats_available() {
		return typeC.get_stats_available();
	}

	// Data Element 2
	/**	RMSE of file's datum relative to absolute datum (x, y, z)
	In same units as indicated by elements 8 and 9 of logical
	record type A.
	@see com.c5corp.c5dem.TypeA#get_planimetric_unit
	@see com.c5corp.c5dem.Dem#get_planimetric_unit
	@see com.c5corp.c5dem.TypeA#get_elevation_unit
	@see com.c5corp.c5dem.Dem#get_elevation_unit
	*/
	public short get_file_rmse() {
		return typeC.get_file_rmse();
	}

	// Data Element 3
	/**	Sample size on which statistics in data element 2 are based
	If 0, then accuracy will be assumed to be estimated rather
	than computed. */
	public short get_file_rmse_sample_size() {
		return typeC.get_file_rmse_sample_size();
	}


	// Data Element 4
	/**	Code indicating availability of statistics in data element 5:
	1=available, 0=unavailable */
	public short get_availability() {
		return typeC.get_availability();
	}

	// Data Element 5
	/** RMSE of DEM data relative to file's datum (x, y, z)
	In same units as indicated by elements 8 and 9 of logical
	record type A.
	@see com.c5corp.c5dem.TypeA#get_planimetric_unit
	@see com.c5corp.c5dem.Dem#get_planimetric_unit
	@see com.c5corp.c5dem.TypeA#get_elevation_unit
	@see com.c5corp.c5dem.Dem#get_elevation_unit*/
	public short get_data_rmse() {
		return typeC.get_data_rmse();
	}

	// Data Element 6
	/**	Sample size on which statistics in element 5 are based
	If 0, then accuracy will be assumed to be estimated rather
	than computed. */
	public short get_data_rmse_sample_size() {
		return typeC.get_data_rmse_sample_size();
	}

	// misc
	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy + ": " + inputfile;
	}

}














