package com.c5corp.DEMconvert.filters;

/** Dem2C5UTM.java  - part of C5UTM database
* C5 DEM Tool 1.0.3b filter that processes a dem to update the C5UTM database (uses java JDBC)
* Dem2C5UTM.java adds UTM points to the C5UTM (C5 Landscape) db.
* Refer to sql directory for info on the data model.
* This version copies data from a dem file directly
* into the database. The writeHeader method updates
* the DEM_METADATA table, then issues a report to
* the 'receipt' file in the RECEIPTS directory or pwd.
* The writeData method updates the UTM_COORDS table
* and similarly reports to the text reciept.
* The db_url and jdbc driver can be changed in the conf/add2c5utm.conf file
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import java.io.*;
import java.util.*;
import java.sql.*;
import com.c5corp.c5dem.*;
import com.c5corp.c5utm.*;
import com.c5corp.DEMconvert.*;

// all filters must be an immediate subclass of C5DemAbstractFilter
public final class Dem2C5UTM extends C5DemAbstractFilter implements C5DemConstants {

	// declarations
	private String in_filename = null; // if running in DemTool, this is moot
	// private String out_filename = null;
	private static String backoutID = null; // probably should not be static
	private static String messages = "";

	// the db_url and driver can be changed in the conf/add2c5utm.conf file
	// be sure to set permissions on that file appropriately!
	//private static String db_url = null;
	//private static String db_driver = null;

	// users
	private static String access_user = "C5UTM_access";
	private static String update_user = "C5UTM_update";

	// confs
	private C5UTMconfs confs;

	// default output dir
	private static String out_dir = "RECEIPTS";

	// various validation checks - all of these must be true for writeHeader or writeData to do anything

	// rez_ok is set to true if the spacial resolution, (horizontal) is 10 meter or 30 meter,
	// all others will be cause writeHeader and writeData to return without updating the db
	// As of this time, all USGS data is 10 meter or 30 meter.
	private boolean rez_ok = false;

	// if interpolate is true, writeData will interplate 10 meter data to 30 meter data
	private boolean interpolate = false;

	// data_validates is set to true if the data in the dem is compatible with the
	// units configured in the database
	private boolean data_validates = false;

	// if writeHeader is able to add a new DEM_METADATA, it sets go to true
	// to let writeData know that it is OK to add the points.
	private boolean go = false;

	// set to true if the configuration file conf/add2c5utm.conf is complete
	// and the database is working, including password configuration.
	private static boolean database_config = false;

	//private DemTable demtab = null;	// DemTable has a 2D array of UtmCoordinatePairElev objects
					// If you only need "straight" DEM data, use a straight DEM object...
					// (DemTable is a child of Dem.)
	private PrintWriter out = null;

	// just trying to keep these references alive when the main method
	// calls the constructor... because I think they are getting collected
	// in_filename and backoutID are always null in the constructor
	private static String filename;
	private static String id;

	// constructors

	/** default constuctor - required by the newInstance() method of the Class class
	* such that this class can be dynamically loaded.
	*/
	public Dem2C5UTM() {
	}

	/** This constructor is typically used by the main method in this class
	* when invoked from the command line.
	*/
	public Dem2C5UTM(String in_fileame, String backoutID) {

		// in_filename and backoutID are always null !
		//System.out.println(in_filename + "\n" + backoutID + "\n" + getOutputPath());

		// System.out.println(filename + "\n" + id + "\n" + getOutputPath());

		//this.backoutID = backoutID;
		this.backoutID = id;

		// create the demTable object
		// demtab = new DemTable(in_filename);
		DemTable demtab = new DemTable(filename);

		// open output file (using parent class method)
		// out = openOutputFile(in_filename, getOutputPath())
		// out = openOutputFile(filename, getOutputPath());

		// write the file
		writeHeader(demtab, out);

		writeData(demtab, out);
	}

	// the main function for command line usage
	public static void main (String args[]) {
		String in = null;
		String backoutID = null;
		String check_me;
		// for checking options...
		boolean found_i = false;
		boolean found_d = false;
		boolean found_b = false;
		messages = "";

		// check the arguments and input validation
		if (args.length == 4 || args.length == 2) {
			for (int i=0; i < args.length; i+=2) {
				if (args[i].equals("-i") && !found_i) {
					in = args[i+1];
					found_i = true;
				} else if (args[i].equals("-d") && !found_d) {
					out_dir = args[i+1];
					found_d = true;
				} else if (args[i].equals("-b") && !found_b) {
					backoutID = args[i+1];
					found_b = true;
				} else {
					usageWarn();
					return;
				}
			}
		} else {
			usageWarn();
			return;
		}

		// find out if we are updating or deleting
		if (found_i && found_b) {
			usageWarn();
			System.err.println("usage issue: -i (input file) and -b (backout id) makes no sense");
			return;
		}

		// make sure the input file is a .dem or a .dat
		if (found_i) {
			check_me = in.substring(in.length()-4, in.length());
			if (!(check_me.equals(".dem"))) {
				usageWarn();
				System.err.println("usage issue: -i input file must be .dem");
				return;
			}
		} else if (found_b) {
			// drop all the data, (except from optional tables)
			System.out.println(C5UTM.deleteDem(backoutID));
			System.out.println("Don't forget to delete the RECIEPT file,");
			System.out.println("or you may have trouble re-adding this dem in the future.");
			return;
		}

		//System.out.println("infile: " + in + " outdir: " + out_dir + " backid: " + backoutID);

		// using a static technique to 'pass' these to the constructor
		filename = in;
		id = backoutID;

		new Dem2C5UTM(in, backoutID);
	}

	/** A method for output of metadata to the file (overrides C5DemAbstractFilter)
	*/
	public void writeHeader(Dem dem, PrintWriter out) {
		// output header
		messages = "";

		// declarations
		double[] adub;

		// set up confs and open files...
		confs = new C5UTMconfs("conf" + File.separatorChar);

		// make sure there is a database and it is working
		database_config = confs.databaseWorking() && confs.databaseUnitsConfigured();

		// check to make sure database_config is ok
		if (!database_config) {
			print("config error: db_url, db_driver, access_password, and update password ");
			print("must be configured in conf/add2c5utm.conf and in the ");
			print("database itself. And the tables must be ready in the DB, of course.");
			println("Dianostics: database working: " + confs.databaseWorking()
				+ ", database units configured: " + confs.databaseUnitsConfigured());

			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeHeader(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		// make sure the file is not already in the DB
		if (!fileAllreadyAdded(dem)) { // if there is no match, we are OK, go ahead
			print("Does not exist in DB, (OK): ");
			print(dem.get_file_name());
		} else { // refuse to add - exit out
			print("Can not add, already in DB: ");
			println(dem.getInputFile());
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeHeader(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		// the following sections are data validation for the input dem file

		// verify that the type A record is UTM, and that the vertical datum
		// and horizontal datum matches the instalation (if not, bail)

		if (dem.get_planimetric_system() == 1 && // utm
			hdatum[dem.get_horizontal_datum()-1].equals(confs.getHorizontalDatum()) &&
				vdatum[dem.get_vertical_datum()-1].equals(confs.getVerticalDatum())) {
			data_validates = true;
		} else {
			println("Not added: C5UTM requires UTM, "
				+ confs.getHorizontalDatum() + ", " + confs.getVerticalDatum());
			println(dem.get_file_name() + "\ncontains " + system[dem.get_planimetric_system()] + ", " +
				hdatum[dem.get_horizontal_datum()-1] + ", " + vdatum[dem.get_vertical_datum()-1]);
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeHeader(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		//) verify that the spacial rez (vertical) is 30 meters (if not, set interpolate to true) - writeHeader()
		if (dem.get_spacial_rez_x() == 10 && dem.get_spacial_rez_y() == 10) {
			interpolate = true;
			rez_ok = true;
		} else if (dem.get_spacial_rez_x() == 30 && dem.get_spacial_rez_y() == 30) {
		// these may later be configurable in conf so that arbitrary rez databases can be created
			// interpolate = false
			rez_ok = true;
		} else {
			println("Input spacial resolution (vertical) x:" +
				dem.get_spacial_rez_x() +
					" y:" + dem.get_spacial_rez_y() +
				"\nis incorrect for C5UTM. (Must be 30, can interpolate 10)");
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeHeader(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		//) update DEM_METADATA - writeHeader()

		double[][] corners=dem.get_ground_coordinates();
		try {
			Connection connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());
			Statement insert = connection.createStatement();

			// insert a new DEM_METADATA row (use dem object accessors etc.)

			// DEM_METADATA_id		SMALLINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY
			// AUTO_INCREMENT will do the work for DEM_METADATA_id
			// zone			TINYINT
			// columns_x		SMALLINT
			// max_profile_length_y	SMALLINT
			// sw_easting		FLOAT
			// sw_northing		FLOAT
			// nw_easting		FLOAT
			// nw_northing		FLOAT
			// ne_easting		FLOAT
			// ne_northing		FLOAT
			// se_easting		FLOAT
			// se_northing		FLOAT
			// date_added		DATETIME
			// local_dem_file	CHAR(40)
			// file_info		CHAR(140)

			String in_file = dem.getInputFile(); // this gets the local file
			in_file = in_file.substring(in_file.lastIndexOf(File.separator)+1);
			// need a DemTable reference for dem to use the average elevation...
			DemTable demtable = (DemTable)dem;

			insert.executeUpdate("INSERT INTO DEM_METADATA (DEM_METADATA_id, zone, columns_x, " +
				"max_profile_length_y, sw_easting, sw_northing, " +
				"nw_easting, nw_northing, ne_easting, ne_northing, se_easting, se_northing, " +
				"date_added, local_dem_file, file_info, average_elevation) " +
				"values (MD5('" + corners[0][0] + corners[0][1] + corners[1][0] + // the corners for an MD5 key
					corners[1][1] + corners[2][0] + corners[2][1] +
					corners[3][0] + corners[3][1]
				+ "'), " +
				dem.get_planimetric_zone() + ", " +
				dem.get_column_count() + ", " +
				dem.maxElevationsForAllProfiles() + ", " +
				corners[0][0] + ", " +
				corners[0][1] + ", " +
				corners[1][0] + ", " +
				corners[1][1] + ", " +
				corners[2][0] + ", " +
				corners[2][1] + ", " +
				corners[3][0] + ", " +
				corners[3][1] + ", " +
				"NOW()" + ", " +
				"'" + in_file + "', " +
				"'" + dem.get_file_name() + "', " + // this is the type A record for the dem
				+ demtable.averageDemElevation() +
				")"
			);

			go = true; // able to add to database
			insert.close();
			connection.close();     // close db connection
			println("\nAdded new DEM_METADATA record...");
		} catch (Exception e) {
			println("Exception on adding new DEM_METADATA\n");
			e.printStackTrace();
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeHeader(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}
	}

	/** A method for output of the formated data (overrides C5DemAbstractFilter) */
	public void writeData(Dem dem, PrintWriter out) {
		// declarations
		TypeB typeB;
		short[] srt;
		int[] elevations;
		double[] adub;
		double conversionVertical = 1.0;
		String demid = "";
		messages = "";
		// double conversionHorizontal = 1.0; // not using this - if someone finds a dem with
		// other than meters for horizontal units, you can impement a conversion.

		//) update UTM_COORDS - writeData()

		// make sure that we are ready to add new points
		if (!go) { // refuse to add - exit out
			print("Can not add points from file: ");
			println(dem.getInputFile());
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeData(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		// check to make sure rez_ok, data_validates, database_config
		if (!(rez_ok && data_validates && database_config)) {
			println("Validation error: cannot write to database: file resolution ok: "
				+ rez_ok + " data_validates (dem and db match): " +
				data_validates + " config_valid: " + database_config);
			out.println(messages);
			// write a receipt
			println("Writing receipt file.");
			ReceiptWriter.writeReceipt(
				confs,
				(DemTable)dem,
				"Dem2C5UTM.writeData(Dem dem, PrintWriter out)",
				messages,
				"(no backout ID, add failed)",
				out
			);
			return;
		}

		// if the vertical units do not match, convert to database conf units

		// System.out.println(confs.getVerticalUnits() + "|" + units[dem.get_elevation_unit()-1]);
		if (!confs.getVerticalUnits().equals(units[dem.get_elevation_unit()-1])) { // if vertical units not the same
			if (dem.get_elevation_unit() == 1) { // feet
				// set conversionVertical to convert feet to meters
				conversionVertical = FEETTOMETERS;
				println("converting vertical feet (input file) to meters (database units)");
			} else if (dem.get_elevation_unit() == 2) { // meters
				conversionVertical = METERSTOFEET;
				println("converting vertical meters (input file) to feet (database units)");
			} else {
				println("This dem might contain arc seconds...");
			}
		}

		// if the horizontal units do not match, convert to database conf units
		// not implemented because this is not likely - all contemporary data seems to be UTM, meters

		// get the new DEM_METADATA_id that was just added to the db
        //try {
		Vector vec = C5UTM.findExactDemNames(dem.get_file_name(), confs, DbHelper.getDbConnection());
		// Vector vec = C5UTM.getDemMetadata(dem.get_file_name());

		// there should only ever be one unless there has
		// been an error adding records to the database
		if (vec.size() < 1) {
			System.err.println("This should never happen.\n" +
				"This code on checks the new DEM_METADATA record that\n" +
				"was just added. Perhaps the access pasword is wrong?\n" +
				"Anyway, serious bad. Good luck fixing.\n");
		} else {
			for (int i=0; i < vec.size(); ++i) {
				demid = ((DemMetadata)(vec.get(0))).getDemId();
				println("Was able to retrieve new DEM_METADATA.DEM_METADATA_id.");
				println("The DEM_METADATA_id is: " + demid);
			}
		}
		/*} catch (SQLException e) {
			System.out.println("Exception on checking the new DEM_METADATA record that\n"
				+ "was just added. Serious problem. Try manually deleting that record.");
					e.printStackTrace();
			return;
		}*/

		// System.out.println("The DEM_METADATA_id is: " + demid);

		// ok, add the points.
		Connection connection = null;

		try {
			connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());
		} catch (SQLException e) {
			System.err.println(e);
		}

		int count=0;
		int increment_adjust = 1; // default is 1
		// increment by 3 if downsample (interpolate) (10 to 30 meter data)
		if (interpolate) {
			increment_adjust = 3;
		}

		// get the array of UtmCoordinatePairElev objects
		// need to cast the dem to a demtable to use getTable()
		DemTable demtab = (DemTable)dem;
		UtmCoordinatePairElev[][] table = demtab.getTable();

		// candidate for C5UTMdatabase methods
		for (int i=0; i<table.length; i=i+increment_adjust) {
			for (int j=0; j<table[i].length; j=j+increment_adjust) {
				if (table[i][j]!=null) {
					int id_temp;

					try {
						Statement insert = connection.createStatement();
						insert.executeUpdate("INSERT INTO UTM_COORD " +
							"(UTM_COORD_id, DEM_METADATA_id, easting, northing, elevation) " + // about to create an MD5 key
							"values ( MD5('" + demtab.get_planimetric_zone() + table[i][j].getEasting() + table[i][j].getNorthing() + "'), '" +
							demid + "'," +
							table[i][j].getEasting()  + "," +
							table[i][j].getNorthing() + "," +
							Math.round(table[i][j].getElevation()*conversionVertical) + ")"
						);
						insert.close();
					} catch (SQLException e) {
						System.err.println("***Exception on adding new UTM_COORDS: " + demid + "\n***" +
						e + "\n"
						+ "*** e: " + table[i][j].getEasting()  + " n:" + table[i][j].getNorthing());
						System.err.println("(Probably overlapping point from adjacent dem. No problem.)");
						continue;
					}
					++count;
				}
			}
		}
		try {
			connection.close();
		} catch (SQLException e) {
			System.err.println("Exception closing connection while processing " + demid + ".\n");
		}

		println("Added " + count + " points to UTM_COORDS with DEM_METADATA_id " + demid + ".");


		// ok, all done. print receipt.
		println("Writing receipt file.");
		println("File added to C5UTM with success");
		ReceiptWriter.writeReceipt(
			confs,
			(DemTable)dem,
			"Dem2C5UTM.writeData(Dem dem, PrintWriter out)",
			messages,
			demid,
			out
		);
	}

	private boolean fileAllreadyAdded(Dem dem) {
		// check that the DEM is not already in the DB
		// exit with and error message if it is
		Vector results;
		String testname = dem.get_file_name();
		//testname = testname.substring(0,testname.lastIndexOf(",")+12); // example: SHINN MOUNTAIN, CA-24000
		System.out.println("Checking if '" + testname + "' already exists in the database...");

		// get a result set from database
		results = C5UTM.findExactDemNames(testname, confs, DbHelper.getDbConnection());

		if (results.size() == 0) {
			return false;
		} else {
			return true;
		}
	}

	private static void usageWarn() {
		System.err.println("usage: Dem2C5UTM (-i infile.dem | -b backoutID) [-d RECEIPT dir]\n" +
					"Check out the README.");
	}

	// the following static functions mirror print and println of System.out
	// and help certain info to be sent to the screen and to a receipt file.
	private static void print(String str) {
		System.out.print(str);
		messages = messages + str;
	}

	private static void println(String str) {
		System.out.println(str);
		messages = messages + str + "\n";
	}

	/** A method that returns a string containing information about the filter.*/
	public String getFilterInfo() {
		return "This filter adds a dem to the C5UTM database. \nThe database must be installed to use it.\n"
			+ com.c5corp.c5dem.C5DemConstants.copy;
	}

	/** Returns a relative path describing the location of the output.*/
	public String getOutputPath() {
		return out_dir;
	}

	/** sets the path to the output file */
	public void setOutputPath(String path) {
		out_dir = path;
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
