package com.c5corp.DEMconvert.filters;

/** Dem2C51KGrid.java  - part of C5 Landscape (C5UTM) database
* <p>Dem2C51KGrid is a C5 DEM Tool filter that processes a dem to
* update the C5UTM database, using java JDBC, and produce image files.</p>
* <p>Dem2C51KGrid adds data to the C5UTM db UTM_1K_GRID_DATA and
* UTM_POINT_STATS tables, copying data from a dem file directly into the
* database, and producing related images that are written to the specified
* web image directory. See the C5UTMconfs class.</p>
* <p>The purpose of this filter is to preprocess into the database as much
* data as is possible directly from the dem files, which is faster than
* getting the same data back out of the database's UTM_COORD table, thus
* helping to build the UTM_1K_GRID_DATA and UTM_POINT_STATS tables in more
* reasonable time.</p>
* <p>Once run over a dem, this filter will set the DEM_METADATA.1K_grid
* attribute to level 1, or "partially processed". Level one indicates that
* the dem's complete 1K areas (at multiples of 1000 in the UTM grid)
* have been processed.</p>
* <p>This filter can be executed as a batch over all available dems
* via the com.c5corp.c5utm.util.Build1KGridFromFiles class (main method).
* The com.c5corp.c5utm.util.Build1KGridFromDb utility should be used to
* process the the overlaping areas shared between dems. (1K areas with
* points in adjacent dems). This filter does not interpolate 10 meter data to
* 30 meter data, thus those dems originally in 10 meter format must be
* fully processed with Build1KGridFromDb.</p>
* <p>The com.c5corp.c5utm.util.Build1KGridStats utility will execute
* these utilities wisely, using the DEM_METADATA.1K_grid as a guide.</p>
* <p>Refer to sql directory for info on the data model.</p>
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0.3
* @see com.c5corp.c5utm.util.Build1KGridFromFiles
* @see com.c5corp.c5utm.util.Build1KGridFromDb
* @see com.c5corp.c5utm.util.Build1KGridStats
* @see com.c5corp.c5utm.C5UTMconfs
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

import java.io.*;
import java.util.*;
import java.sql.*;
import com.c5corp.c5dem.*;
import com.c5corp.c5utm.*;
import com.c5corp.DEMconvert.*;

// all filters must be an immediate subclass of C5DemAbstractFilter
public final class Dem2C51KGrid extends C5DemAbstractFilter implements C5DemConstants {

	// declarations
	private String in_filename = null; // if running in DemTool, this is moot

	// the db_url and driver can be changed in the com/c5corp/c5utm/conf/add2c5utm.conf file
	// be sure to set permissions on that file appropriately!
	// private static String db_url = null;
	// private static String db_driver = null;

	// confs
	private C5UTMconfs confs;

	// default output dir
	private static String out_dir = "RECEIPTS";

	// various validation checks - all of these must be true for writeHeader or writeData to do anything

	// rez_ok is set to true if the spacial resolution, (horizontal) 30 meter,
	// any other will be cause writeHeader and writeData to return without updating the db
	// As of this time, all USGS data is 10 meter or 30 meter.
	private boolean rez_ok = false;

	// data_validates is set to true if the data in the dem is compatible with the
	// units configured in the database
	private boolean data_validates = false;

	// set to true if the configuration file com/c5corp/c5utm/conf/add2c5utm.conf is complete
	// and the database is working, including password configuration.
	private static boolean database_config = false;

	//private DemTable demtab = null;	// DemTable has a 2D array of UtmCoordinatePairElev objects
					// If you only need "straight" DEM data, use a straight DEM object...
					// (DemTable is a child of Dem.)
	private PrintWriter out = null;

	//Connection connection = null;

	// constructors

	/** Default constuctor, required by the newInstance() method of the Class class,
	* such that this class can be dynamically loaded.
	*/
	public Dem2C51KGrid() {
		//connection = DbHelper.getDbConnection(); // this class reuses this, must close it later.
	}

	/** Builds tables from the specified file name.
	* Can be invoked from the command line.
	*/
	public Dem2C51KGrid(String filename) {
		this.in_filename = filename;

		//connection = DbHelper.getDbConnection(); // this class reuses this, must close it later.

		// create the demTable object
		// demtab = new DemTable(in_filename);
		DemTable demtab = new DemTable(filename);

		// open output file (using parent class method)
		//out=openOutputFile(demtab.get_file_name(), getOutputPath());
		// this may seem strange, to pass a null reference below. This
		// is because we have to implement writeHeader and writeData
		// as specified in C5DemAbstractFilter, which assumes file output.
		// In this unique case, we write to a database, not a file.

		// write the file
		writeHeader(demtab, out);

		writeData(demtab, out);
	}

	// the main function for command line usage
	public static void main (String args[]) {
		String in = null;
		String check_me;
		// for checking options...
		boolean found_i = false;
		boolean found_d = false;

		// check the arguments and input validation
		if (args.length == 4 || args.length == 2) {
			for (int i=0; i < args.length; i+=2) {
				if (args[i].equals("-i") && !found_i) {
					in = args[i+1];
					found_i = true;
				} else if (args[i].equals("-d") && !found_d) {
					out_dir = args[i+1];
					found_d = true;
				} else {
					usageWarn();
					return;
				}
			}
		} else {
			usageWarn();
			return;
		}

		// make sure the input file is a .dem
		if (found_i) {
			check_me = in.substring(in.length()-4, in.length());
			if (!(check_me.equals(".dem"))) {
				usageWarn();
				System.err.println("usage issue: -i input file must be .dem");
				return;
			}
		}

		new Dem2C51KGrid(in);
	}

	public void writeHeader(Dem dem, PrintWriter out) {

    	// declarations
		// set up confs and open files...
		confs = new C5UTMconfs();

		// make sure there is a database and it is working
		database_config = confs.databaseWorking() && confs.databaseUnitsConfigured();

		// check to make sure database_config is ok
		if (!database_config) {
			System.err.print("config error: db_url, db_driver, access_password, and update password ");
			System.err.print("must be configured in com/c5corp/c5utm/conf/add2c5utm.conf and in the ");
			System.err.println("database itself. And the tables must be ready in the DB, of course.");
			System.err.println("configuration messages: " + confs.getMessages());
			return;
		}

		// the following sections are data validation for the input dem file

		// verify that the type A record is UTM, and that the vertical datum
		// and horizontal datum matches the installation (if not, bail)

		if (dem.get_planimetric_system() == 1 && // utm
			hdatum[dem.get_horizontal_datum()-1].equals(confs.getHorizontalDatum()) &&
				vdatum[dem.get_vertical_datum()-1].equals(confs.getVerticalDatum())) {
			data_validates = true;
		} else {
			System.err.println("Not processed: this C5UTM installation requires data in UTM coords, the horizontal datum "
				+ confs.getHorizontalDatum() + ", and the vertical datum " + confs.getVerticalDatum() + ".");
			System.err.println(dem.get_file_name() + "\ncontains " + system[dem.get_planimetric_system()] + ", " +
				hdatum[dem.get_horizontal_datum()-1] + ", " + vdatum[dem.get_vertical_datum()-1]);
			// data_validates = false;
			return;
		}

		//) verify that the spacial rez (vertical) is 30 meters
		if (dem.get_spacial_rez_x() == 30 && dem.get_spacial_rez_y() == 30) {
		// these may later be configurable in conf so that arbitrary rez databases can be created
			rez_ok = true;
		} else {
			System.err.println("Warn: can't process: " + dem.get_file_name() +
				": The input spacial resolution (ing meters) x:" +
					dem.get_spacial_rez_x() +
						" y:" + dem.get_spacial_rez_y() +
							"\ncan not be converted with Dem2C51KGrid. (Both must be 30).\nBut Build1KGridFromDb can do it " +
								"Because the dem data is downsampled to 30 meters when added to the database.");
			// rez_ok = false;
			return;
		}
	}

	/** A method to write the output data to the (database). (overrides C5DemAbstractFilter) */
	public void writeData(Dem dem, PrintWriter out) {

		// cast the dem into a DemTable (its real type)
		DemTable demtab = (DemTable)dem;

		// booleans to convert vertical units back
		boolean feetConvertedToMeters = false;
		boolean metersConvertedToFeet = false;

		// check the condtion of the database

		// check to make sure rez_ok, data_validates, database_config
		if (!(rez_ok && data_validates && database_config)) {
			System.err.println("Validation error: cannot write to database: file resolution ok: " +
				rez_ok + " data_validates (dem and db match): " +
					data_validates + " config_valid: " + database_config +
						" for file: " + dem.get_file_name() + ", (all must be true).");
			return;
		}

		// if the vertical units do not match, convert to database conf units
		if (!confs.getVerticalUnits().equals(units[dem.get_elevation_unit()-1])) { // if vertical units not the same
			if (dem.get_elevation_unit() == 1) { // feet
				//  convert feet to meters
				demtab.convertToVerticalMeters();
				feetConvertedToMeters = true;
			} else if (dem.get_elevation_unit() == 2) { // meters
				demtab.convertToVerticalFeet();
				metersConvertedToFeet = true;
			} else {
				System.err.println(dem.get_file_name() + " might contain an unknown vertical unit.");
				return;
			}
		}

		// if the horizontal units do not match, convert to database conf units
		// not implemented because this is not likely - all contemporary data seems to be UTM, meters

		// make sure that this data has not already been added
		// make sure the dem file *is* already in the DB
		Connection con = DbHelper.getDbConnection();
		if (dataOKtoAdd(dem, con)) { // if there is a match for the dem, but not for grid data, we are OK, go ahead
			System.out.print("Dem2C51KGrid will try to add the data from:");
			System.out.println(dem.getInputFile());
		} else { // refuse to add - exit out
			System.err.print("Dem2C51KGrid: either the 1K grid data already exists in the DB, or the dem data has not been added: ");
			System.err.println(dem.get_file_name());
			return;
		}
		DbHelper.closeDbConnection(con);

		// iterate over each possible full 1k by 1k (in 1K multiples) area in the demtable
			// calculate the stats for each
			// insert into UTM_POINT_STATS
			// get the new id for that record
			// insert into UTM_1K_GRID_DATA corresponding to UTM_COORDS
		// figure out the sw coords in the dem that snap near multiples of 1000
		// how far should we iterate until we are sure we have exited
		// the area represented by the dem?
		PreciseUTMcoordPair sw = demtab.getSWcorner();
		PreciseUTMcoordPair nw = demtab.getNWcorner();
		PreciseUTMcoordPair ne = demtab.getNEcorner();
		PreciseUTMcoordPair se = demtab.getSEcorner();

		// lesser of the sw and nw easting
		double temp = sw.getEasting() < nw.getEasting()?sw.getEasting():nw.getEasting();
		int easting_start = ((int)temp/1000)*1000;
		// lesser of the sw and se northing
		temp = sw.getNorthing() < se.getNorthing()?sw.getNorthing():se.getNorthing();
		int northing_start = ((int)temp/1000)*1000;


		// greater of the ne and se easting
		temp = ne.getEasting() > se.getEasting()?ne.getEasting():se.getEasting();
		int easting_max = (((int)temp/1000)*1000+1000)-easting_start;
		// greater of the nw and ne northing
		temp = nw.getNorthing() > ne.getNorthing()?nw.getNorthing():ne.getNorthing();
		int northing_max = (((int)temp/1000)*1000+1000)-northing_start;		// insert the records...

		Point any = null;

		try {
			Connection connection=DriverManager.getConnection(confs.getDbUrl(),
				"C5UTM_update", confs.getUpdatePassword());

			for (int x = 0; x < easting_max; x+=1000) {
				for (int y = 0; y < northing_max; y+=1000) {
					UtmCoordinatePairElev[][] area = demtab.getPoints(x + easting_start, y + northing_start, 1000, 1000);
					if (area != null) { // if returned null, we are at a boundary and wish to do nothing
						addToDatabase(area, dem, connection);
						createImage(area, dem, x + easting_start, y + northing_start);
					}
				}
			}

			// go to the database for the point representing any point in the DB from this dem
			// (it will contain the DEM_METADATA_id and the id)
			any = C5UTM.getPoint(dem.get_planimetric_zone(),
				easting_start+2000,
				northing_start+2000,
				connection);

			Statement insert = connection.createStatement();

			// update DEM_METADATA
			insert = connection.createStatement();
			insert.executeUpdate("UPDATE DEM_METADATA SET 1K_grid=1 WHERE DEM_METADATA_id=\"" + any.getDemId() + "\"");
			insert.close();

			connection.close();
		} catch (SQLException e){
			System.err.println("Exception UPDATE DEM_METADATA: " + any.getDemId() + "\n");
            e.printStackTrace();
		}

		// Change the units of the dem back (this is in case we are using the DemTool -
		// changing the units will fubar the DemTool image panel.
		if (feetConvertedToMeters == true) {
			demtab.convertToVerticalFeet();
		} else if (metersConvertedToFeet == true) {
			demtab.convertToVerticalMeters();
		}

		System.out.println("Succesfully processed 1K grid data for " + in_filename + " to level 1.\n");
	}

	private void createImage(UtmCoordinatePairElev[][] area, Dem dem, int easting, int northing) {

		//build a file name
		int zone=dem.get_planimetric_zone();
		//int easting = area[0][0].getEasting(); 	// instead of using the exact corner vals
		//int northing = area[0][0].getNorthing(); 	// using the closest utm grid multiple of 1000
													// which is passed in from the for loop calling this method.
		String east = new String("" + easting).substring(0,3);
		String north = new String("" + northing).substring(0,4);

		String fileName = new String(zone + "_" + east + "_" + north);

		// create file and check if the file exists
		File file = new File(confs.getUTMImageDir() + File.separator + fileName);
		if (!file.exists()) {
			// build an artificial point
			Points points = ArtificialPoints.getPoints(area, dem.get_planimetric_zone());

			// write the file
 			UtmImage image = new UtmImage(points,
				UtmImage.RENDER_HIGH |
				UtmImage.RENDER_LOW |
				UtmImage.RENDER_MEAN |
				UtmImage.RENDER_MEDIAN |
				UtmImage.RENDER_MODE
			);

			image.writeImageFile(confs.getUTMImageDir() , fileName);
		}
	}

	private boolean addToDatabase(UtmCoordinatePairElev[][] area, Dem dem, Connection connection) {

		Points ob =  ArtificialPoints.getPoints(area, dem.get_planimetric_zone());

		return insertRecord(ob, dem, connection);
	}

	private boolean insertRecord(Points ob, Dem dem, Connection connection) {

		boolean ok = false;

		try {
			// go to the database for the point representing the sw corner (it will contain the DEM_METADATA_id and the id)
			Point sw = C5UTM.getPoint(ob.getPointAt(0,0).getZone(),
				ob.getPointAt(0,0).getEasting(),
				ob.getPointAt(0,0).getNorthing(),
				connection);
			// go to the database for the point representing the 'middle' point (it will contain the DEM_METADATA_id and the id)
			// again, assuming a 34x34 array
			Point mid = C5UTM.getPoint(ob.getPointAt(16,16).getZone(),
				ob.getPointAt(16,16).getEasting(),
				ob.getPointAt(16,16).getNorthing(),
				connection);

			// if we did not get one of these, then we can't do it from the file
			if (mid == null || ob == null) {
				return ok; // false
			}

			// insert the records...
			// first UTM_POINT_STATS

			Statement insert = connection.createStatement();

			try {
				insert.executeUpdate("INSERT INTO UTM_POINT_STATS " +
						"(UTM_POINT_STATS_id, low, high, mean, median, stdev, percentile, " +
						"contiguous_modality_percentage, topographic_descriptor, " +
						" terrain_descriptor) " +
						"values (\"" +
						mid.getId() + "\"," + // the id of the middle point
						ob.getLowest()+ "," +
						ob.getHighest() + "," +
						(int)Math.round(ob.calculateMean()) + "," +
						ob.calculateMedian() + "," +
						ob.calculateStandardDeviation()  + "," +
						ob.calculatePercentile(mid.getElevation()) + "," +
						ob.calculateContiguousModalityPercentage() + "," +
						ob.calculateTopographyDescriptor() + "," +
						ob.calculateTerrainDescriptor() + ")");
				insert.close();

				// then the MODE table
				// build the mode string... INSERT INTO MODE (utm_point_stat_id, mode) VALUES ($pstat_id, $_)
				int[] mode = ob.calculateMode();
				for (int i = 0; i < mode.length; i++) {
					insert = connection.createStatement();
					insert.execute("INSERT INTO MODE (UTM_POINT_STATS_id, mode) VALUES (\"" +
							mid.getId() + "\"," + // the id of the middle point
							mode[i] +
						")");
					insert.close();
				}
			} catch (SQLException e) { // do nothing!
				//System.out.println("No worries if it already exists...");
			}

			// then link to UTM_1K_GRID_DATA
			insert = connection.createStatement();
			insert.executeUpdate("INSERT INTO UTM_1K_GRID_DATA (UTM_1K_GRID_DATA_id, UTM_POINT_STATS_id) VALUES (\"" +
				sw.getId() + "\",\"" +
				mid.getId() +
				"\")");
			insert.close();

			ok = true;
		} catch (SQLException e) {
			System.err.println("Exception on adding new UTM_POINT_STATS,UTM_1K_GRID_DATA, and MODE:: " + ob + "\n");
            e.printStackTrace();
			ok = false;
		}

		return ok;
	}

	private boolean dataOKtoAdd(Dem dem, Connection connection) {
		// check that the DEM is already in the DB
		// and that the 1K grid data is not.
		Vector results;
		String testname = dem.get_file_name();
		boolean gridDataInDb = false;
		boolean demInDb = false;

		System.out.println("Checking if '" + testname + "' is ready to be processed into the 1K grid");

		// get a result set from database
		results = C5UTM.findExactDemNames(testname, confs, connection);

		if (results.size() > 0 ) {
			demInDb = true;
		} else {
			return false;
		}

		DemMetadata metadata = (DemMetadata)results.get(0); // there should only be one result

/*
		// testing here
		System.out.println(in_filename+"*******");

		int temp =  metadata.calculateSurroundingDemAvailability(connection);
		if ((DemMetadata.NW_BIT & temp) == 128) {System.out.println("NW");}
		if ((DemMetadata.W_BIT & temp) == 64) {System.out.println("W");}
		if ((DemMetadata.SW_BIT & temp) == 32) {System.out.println("SW");}
		if ((DemMetadata.S_BIT & temp) == 16) {System.out.println("S");}
		if ((DemMetadata.SE_BIT & temp) == 8) {System.out.println("SE");}
		if ((DemMetadata.E_BIT & temp) == 4) {System.out.println("E");}
		if ((DemMetadata.NE_BIT & temp) == 2) {System.out.println("NE");}
		if ((DemMetadata.N_BIT & temp) == 1) {System.out.println("N");}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.NW_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.W_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.SW_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.S_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.SE_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.E_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.NE_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		try {
			System.out.println((metadata.getAdjacentDemMetadata(DemMetadata.N_BIT, connection)).getLocalDemFile());
		} catch (NullPointerException e) {
			// do nothing
		}

		System.out.println("*******");
*/

		if (metadata.get1kGridStatus() > 0) { // if it has alread been set to partially processed (1) or greater
			gridDataInDb = true;
		} else {
			gridDataInDb = false;
		}

		return demInDb && (!gridDataInDb); // true if dem in db and grid data *not* in db
	}

	private static void usageWarn() {
		System.err.println("usage: Dem2C51KGrid.java -i infile.dem [-d RECEIPT dir]\n" +
					"Check out the README.");
	}

	/** A method that returns a string containing information about the filter.*/
	public String getFilterInfo() {
		return "This filter is a wolly little adventure in data management, building the 1K grid data. \nThe database must be installed to use it.\n. (c) C5 corp, 2002-2003.";
	}

	/** Returns a relative path describing the location of the output.*/
	public String getOutputPath() {
		return out_dir;
	}

	/** sets the path to the output file */
	public void setOutputPath(String path) {
		out_dir = path;
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
