// Converts the elevations in a DEM into a corpscon batch file
/**
* <p>Converts a DEM file into a corpscon batch file - also functions as a filter for DemTool
* Adds X and Y values in the Point Name field.</p>
* in the C5 DEM Tools UI
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
* <p>Current USGS DEM data typically contains data cast on the North American Horizontal
* Datum of 1927, (NAD27), and the vertical datum NGVD29. Some of the newer data is NAD83.</p>
*
* <p>The US army corps of engineers makes a dos/win program called CORPSCON (which
* encapsulates NADCON from noaa for its horizontal conversions), that converts between
* various horizontal and vertical datum. This class exports DEM data into a CORPCON
* batch file to preform such conversions.</p>
*
* <p>CORPSCON is available at: http://crunch.tec.army.mil/software/corpscon/corpscon.html</p>
*
* <p>Thank you to the corps of engineers and noaa.</p>
*
* <p>The following is from the Corpswin.doc (CORPSCON documentation), about the file format
* for UTM grid coords:</p>
*
* <p>For grid coordinates, the fields are:
*
* <ol>
* <li>Point Name</li>
* <li>Easting or X value</li>
* <li>Northing or Y value</li>
* <li>Elevation (optional)</li>
* </ol>
* </p>
* <p>Again, the fourth field is required only if vertical conversions are to be performed.
* Below are some examples of valid lines for grid coordinates:<br>
* <br>
* Point A, 500000.0, 2600000.0, 1238.4566<br>
* , 500010, 2600400.264, 10.66<br>
* PTC, 512353.23523,2600234.23453, 100.6<br></p>
*
* <p>Lines beginning with a ';' or '#' characters in Corpscon Batch Files
* are interpreted as comment lines.</p>
*/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

package com.c5corp.DEMconvert.filters;

import java.io.*;
import com.c5corp.c5dem.*;
import com.c5corp.DEMconvert.*;

public final class CorpsconBatchXY extends C5DemAbstractFilter {

	// declarations
	String in_filename=null; // if running in DemTool, this is moot
	String out_filename=null;
	static String out_dir="DAT"; // default when used as DemTool filter
	DemTable demtab=null;
	PrintWriter out=null;

	// constructors

	/** The default constructor creates a blank object such that
	<code>writeHeader(Dem, PrintWriter)</code> and <code>writeData(Dem, Printwriter)</code>
	can be used with arbitrary Dem and PrintWriter objects.*/
	public CorpsconBatchXY() {
		// just default...
	}

	/** This constructor is used by the main function in this class,
	which is intended mostly for command line use. */
	public CorpsconBatchXY(String in_fileame) {

		// create the demTable object
		demtab= new DemTable(in_fileame);

		// open output file (using parent class method)
		out=openOutputFile(demtab.get_file_name(), getOutputPath());

		// write the file
		writeHeader(demtab, out);

		writeData(demtab, out);

		// close the output file
		out.close();
	}

	/** The main method is maintained so that this filter can be used
	from the command line, in addition to being used inside of c5 dem tool.*/
	public static void main (String args[]) {
		String in;
		try {
			in = args[0];
		} catch (ArrayIndexOutOfBoundsException e) {
			System.err.println("Usage: $ dem2corpsconxy infile.dem [output dir]");
			return;
		}

		try {
			out_dir = args[1];
		} catch (ArrayIndexOutOfBoundsException e) {
			// default for command line use .
			out_dir = ".";
		}

		new CorpsconBatchXY(in.toString()); 	// Without using toString, in_filename is null in
							// the constructor. Mystery.
	}


	/** A method for output of metadata to the file (overrides C5DemAbstractFilter)
	system[], units[], vdatum[] and hdatum[] are from C5DemConstants interface.
	@see com.c5corp.c5dem.C5DemConstants
	*/
	public void writeHeader(Dem dem, PrintWriter theout) {
    	// output header
    	theout.println("#DEM Converted by " + this.toString() + " output filter, http://www.c5corp.com");
		theout.println("#DEM type A record 1: "+ dem.get_file_name());
		theout.println(";SOFTWARE: C5 " + this.toString() + " output filter, part of C5 DEM Tool 1.0 (Java 1.3.1)");
		theout.println(";Horizontal Datum:\t" + system[dem.get_planimetric_system()]
			+ ", " +hdatum[dem.get_horizontal_datum()-1]);
		theout.println(";Horizontal Zone:\t" + dem.get_planimetric_zone());
		if (dem.get_spacial_rez_z() == 0.1) {
			theout.println(";Horizontal Units:\t" + units[dem.get_planimetric_unit()-1] + "(converted decimeters)");
		} else {
			theout.println(";Horizontal Units:\t" + units[dem.get_planimetric_unit()-1]);
		}
		theout.println(";Vertical Datum:\t" + vdatum[dem.get_vertical_datum()-1]);
		theout.println(";Vertical Units:\t" + units[dem.get_elevation_unit()-1]);

	}

	/** A method to write the output (formated) data to the file. (overrides C5DemAbstractFilter) */
	public void writeData(Dem dem, PrintWriter theout){
		// declarations

		// in DEM2corpscon we are using a subclass of dem, DemTable,
		// so it needs to be cast
		DemTable demtab = (DemTable)dem;

		// get the array of UtmCoordinatePair objects

		UtmCoordinatePairElev[][] table = demtab.getTable();

		// output data
		for (int i=0; i<table.length; i++) {
			for (int j=0; j<table[i].length; j++) {
				//
				if (table[i][j]!=null) {
					theout.println(i + " " + j + ",\t" + table[i][j].getEasting() + ", " +
						table[i][j].getNorthing() + ", " +
						table[i][j].getElevation());
				}
			}
		}
	}

	/** A method that returns a string containing information about the filter.*/
	public String getFilterInfo() {
		return "This filter will export dem data into a Corpscon file with X and Y values in the Point Name field.\n";
	}

	/**Returns a relative path describing the location of the output. */
	public String getOutputPath() {
		return out_dir;
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
