/**
* An abstract class implementing c5demOutFilterInterface,
* that can be implemented to create new c5 DemTool filters for exporting
* DEM data into any text format.
* Extend this class to create new output filters that will work in
* the C5 DEM Tool.
* <p>extensions must implement methods:</p>
* <ul>
* 	<li><code>public void writeHeader(Dem dem, PrintWriter out);</code></li>
* 	<li><code>public void writeData(Dem dem, PrintWriter out);</code</li>
* 	<li><code>public String getFilterInfo();</code</li>
* 	<li><code>public String getOutputPath();</code</li>
* </ul>
* The class provides a static method,
* <code>public static PrintWriter openOutputFile(String out_filename, String path)</code>,
* for opening an output file.
* @author Brett Stalbaum copyright 2002-2005
* @version 1.0.3
* @since 1.0
**/

/*
* This library is free software; you can redistribute it and/or modify it under the
* terms of the GNU Lesser General Public License as published by the Free Software
* Foundation; either version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
* without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* See the GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License along with this
* library; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
* Please refer to LICENSE.txt which is distributed with the distribution for the
* full text of the GNU Lesser General Public License
*/

package com.c5corp.DEMconvert;

import java.io.*;
import com.c5corp.c5dem.*;

public abstract class C5DemAbstractFilter
		implements C5DemConstants {

	// define abstract methods for children to override

	/** A method for output of metadata
	* @param dem A Dem.
	* @param out Where to write the data.
	*/
	public abstract void writeHeader(Dem dem, PrintWriter out);

	/** method for output of the formated data
	* @param dem A Dem.
	* @param out Where to write the data.
	*/
	public abstract void writeData(Dem dem, PrintWriter out);

	/** A method that returns a string containing information about the filter.
	@return the string info about the filter
	*/
	public abstract String getFilterInfo();


	/** A method to open a new output file with a name derived from the
	type a metadata, record 1, which contains the map name */
	public static PrintWriter openOutputFile(String out_filename, String path) {

		PrintWriter out;

		// make sure the path has a trailing /
		if (path != "") {
			if (path.charAt(path.length()-1) != '/') path = path + "/";
		}

    		try {
			out_filename= out_filename.substring(0,out_filename.lastIndexOf(","));
			out_filename= out_filename.replace(' ','_');
			out_filename= path + out_filename + ".dat";
    			out = new PrintWriter(new FileOutputStream(out_filename));

		} catch (FileNotFoundException e) {
			System.err.println(e);
			return null;
		}

		System.out.println("writing: " + out_filename);

		return out;
	}


	/** Returns a relative path describing where to place the output the default implementation is to use the present working directory.*/
	public String getOutputPath() {
		return ".";
	}

	/** overrides java.lang.Object.toString() */
	public String toString() {
		return this.getClass().getName() + C5DemConstants.copy;
	}
}
